# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json
import urllib
from datetime import date, timedelta

from dateutil.relativedelta import relativedelta
from six import text_type
from yt.wrapper import YtClient

from travel.avia.library.python.references.partner import PartnerCache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import ClickaviaAviaOrderSnapshot, OrderCurrencyCode, OrderStatus

LOG = get_logger(__name__)


class ClickaviaCollector(HttpCollector):
    PARTNER_NAME = 'clickavia'
    PARTNER_CODE = 'clickaviatravelab'
    BASE_URL = 'https://www.clickavia.ru/api/partners'
    REQUEST_TIMEOUT = 60
    STATUS_MAPPING = {
        'processing': OrderStatus.PENDING,
        'confirmed': OrderStatus.CONFIRMED,
        'paid': OrderStatus.CONFIRMED,
        'canceled': OrderStatus.CANCELLED,
    }

    def __init__(self, options):
        super(ClickaviaCollector, self).__init__(options)

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(self.PARTNER_CODE)

        self.get_day_report = with_retries(
            self.get_day_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        self.base_url = self.base_url + '/{token}/orders'.format(token=urllib.quote(options.token))

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('--token', default=None)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-4)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

    def _get_snapshots(self):
        for day_date in iter_day(self.date_from, self.date_to):
            LOG.info('Getting snapshots for %r', day_date)
            for snapshot in self.get_day_snapshots(day_date):
                if snapshot is None:
                    continue
                yield snapshot

    def get_day_snapshots(self, day_date):
        r = self.get_day_report(day_date)
        for snapshot in self.parse_report(r.content):
            yield snapshot

    def get_day_report_once(self, day_date):
        date_from = day_date - timedelta(days=1)
        date_to = day_date + timedelta(days=1)
        params = {
            'date_from': date_from.strftime('%Y-%m-%d'),
            'date_to': date_to.strftime('%Y-%m-%d'),
        }
        rsp = self.request_get(self.base_url, params=params, timeout=self.REQUEST_TIMEOUT)
        return rsp

    def parse_report(self, content):
        try:
            orders = json.loads(content)
        except ValueError:
            LOG.exception('Bad clickavia JSON. Response starts with "%s"', content[:40])
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        if not isinstance(orders, list):
            return

        for order in orders:
            (airport_from, airport_to), trip_type = self.get_airports(text_type(order['direction']).split('/'))

            snapshot = ClickaviaAviaOrderSnapshot()
            snapshot.update_partner_order_id(text_type(order['id']))

            snapshot.status = self.STATUS_MAPPING[order['state']]
            snapshot.label = text_type(order['subagent'])
            snapshot.created_at = timestamp(parse_datetime_iso(order['created_at']))
            snapshot.order_amount = float(order['sum'])
            snapshot.currency_code = OrderCurrencyCode(order['currency'])
            snapshot.origin = airport_from
            snapshot.destination = airport_to
            snapshot.trip_type = trip_type
            snapshot.partner_id = self.partner_id
            snapshot.billing_order_id = self.billing_order_id

            yield snapshot

    @staticmethod
    def get_airports(directions):
        airport_from = directions[0]
        airport_to = directions[len(directions) / 2]

        trip_type = 'oneway'
        if len(directions) > 2:
            if directions[0] == directions[-1]:
                trip_type = 'roundtrip'
            else:
                trip_type = 'openjaw'

        return (airport_from, airport_to), trip_type
