# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json
from datetime import date, datetime

from dateutil.relativedelta import relativedelta
from yt.wrapper import YtClient

from travel.avia.library.python.marker_helpers import AviaFlightRouteHelper
from travel.avia.library.python.references.partner import PartnerCache
from travel.avia.library.python.references.station import create_station_cache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderStatus, FlyoneAviaOrderSnapshot, OrderCurrencyCode

LOG = get_logger(__name__)


class FlyoneCollector(HttpCollector):
    SOURCE = 'flyone'
    PARTNER_CODE = 'flyone'
    PARTNER_NAME = 'flyone'
    BASE_URL = 'http://reportapi.flyone.eu/api/statistics'
    REQUEST_TIMEOUT = 90
    STATUS_MAPPING = {
        'CANCELLED': OrderStatus.CANCELLED,
        'FAILED': OrderStatus.CANCELLED,
        'PAID': OrderStatus.CONFIRMED,
    }

    def __init__(self, options):
        super(FlyoneCollector, self).__init__(options)

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        self.username = options.username
        self.password = options.password

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(self.PARTNER_CODE)

        self.station = create_station_cache(yt_client)
        self.route_helper = AviaFlightRouteHelper(self.station)

        self.get_day_report = with_retries(
            func=self._get_day_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('-u', '--username', required=True)
        parser.add_argument('-p', '--password', required=True)

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-4)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

    def _get_snapshots(self):
        for day_date in iter_day(self.date_from, self.date_to):
            LOG.info('Getting snapshots for %r', day_date)
            for snapshot in self.get_day_snapshots(day_date):
                if snapshot is None:
                    continue
                yield snapshot

    def get_day_snapshots(self, day_date):
        """
        Собираем данные от партнера и раскладываем в snapshot
        """
        r = self.get_day_report(day_date)
        for snapshot in self.parse_report(r.content):
            yield snapshot

    def _get_day_report_once(self, day_date):
        params = {
            'date1': day_date.strftime('%Y-%m-%d'),
            'date2': day_date.strftime('%Y-%m-%d'),
            'partner': self.username,
            'password': self.password,
        }
        LOG.info(
            'Getting data from %s date: %s',
            self.base_url,
            day_date,
        )
        r = self.request_get(
            self.base_url,
            params=params,
            timeout=self.REQUEST_TIMEOUT,
        )
        return r

    def parse_report(self, content):
        try:
            response = json.loads(content)
            bookings = response['bookings']['booking'] if response.get('bookings') else []
        except:
            LOG.exception('Bad Flyone JSON')
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        for booking in bookings:
            try:
                if not booking:
                    continue
                flight_info = self._parse_flight_info(booking)
                snapshot = FlyoneAviaOrderSnapshot.from_dict(
                    d={
                        'status': self._parse_status(booking.get('state')),
                        'label': booking.get('marker'),
                        'created_at': timestamp(self._convert_dt(booking.get('created_at'))),
                        'order_amount': float(booking.get('price')),
                        'currency_code': OrderCurrencyCode(booking.get('currency')),
                        'origin': flight_info.get('origin'),
                        'destination': flight_info.get('destination'),
                        'trip_type': flight_info.get('trip_type'),
                        'date_forward': flight_info.get('date_forward'),
                        'date_backward': flight_info.get('date_backward'),
                        'source': self.SOURCE,
                        'partner_id': self.partner_id,
                        'billing_order_id': self.billing_order_id,
                    },
                    ignore_unknown=True,
                    convert_type=False,
                )
                snapshot.update_partner_order_id(booking.get('pnr'))
                yield snapshot
            except Exception:
                LOG.exception('Flyone parse error')

    def _parse_status(self, partner_status):
        return self.STATUS_MAPPING[partner_status]

    def _parse_flight_info(self, booking):
        try:
            segments = [
                {
                    'from': flight.get('departure'),
                    'to': flight.get('arrival'),
                    'departure_dt': datetime.strptime(
                        '{} {}'.format(flight.get('departureDate'), flight.get('departureTime')),
                        '%Y-%m-%d %H:%M',
                    ),
                    'arrival_dt': datetime.strptime(
                        '{} {}'.format(flight.get('arrivalDate'), flight.get('arrivalTime')),
                        '%Y-%m-%d %H:%M',
                    ),
                }
                for flight in booking['segment']['flight']
            ]
        except ValueError:
            segments = []
        flight_info = {'flights': segments}
        if segments:
            try:
                return self.route_helper.fillin_trip_info(flight_info)
            except Exception:
                LOG.exception('Cannot fill airports for order. %s', segments)
                raise ProcessError(ErrorType.ET_PARTNER_DATA)
        return {}

    @staticmethod
    def _convert_dt(value):
        return datetime.strptime(value, '%Y-%m-%d %H:%M:%S') if value else None
