from __future__ import unicode_literals

from travel.cpa.collectors.lib.yandex_avia_collector import YandexAviaCollector
from travel.cpa.lib.lib_datetime import parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import KupibiletAviaOrderSnapshot, OrderCurrencyCode, OrderStatus

LOG = get_logger(__name__)


class KupibiletCollector(YandexAviaCollector):
    PARTNER_NAME = 'kupibilet'
    PARTNER_CODE = 'kupibilet'

    BASE_URL = 'https://api.kupibilet.ru/5991d8dd414d502bd12c71de3f364179'

    SNAPSHOT_CLS = KupibiletAviaOrderSnapshot
    BOOKINGS_PATH = '//bookings/booking'
    FILL_AIRPORTS = False

    STATUS_MAPPING = {
        'booked': OrderStatus.PENDING,
        'paid': OrderStatus.CONFIRMED,
        'cancelled': OrderStatus.CANCELLED,
    }

    def __init__(self, options):
        super(KupibiletCollector, self).__init__(options)
        self.login = options.login
        self.password = options.password

        self.params = dict(
            partner=options.login,
            password=options.password,
        )

    @classmethod
    def configure(cls, parser):
        YandexAviaCollector.configure(parser)
        parser.add_argument('--login')
        parser.add_argument('--password')
        parser.add_argument('--base-url', default=cls.BASE_URL)

    def get_order_snapshot(self, raw_snapshot):
        snapshot_dict = self.get_node_dict(raw_snapshot)

        snapshot = self.SNAPSHOT_CLS.from_dict(dict(), ignore_unknown=True)

        if snapshot_dict.get('is_roundtrip'):
            if 'true' == snapshot_dict.get('is_roundtrip'):
                snapshot.trip_type = 'roundtrip'
            elif 'false' == snapshot_dict.get('is_roundtrip'):
                snapshot.trip_type = 'oneway'

        snapshot.update_partner_order_id(snapshot_dict['id'])
        snapshot.status = self.STATUS_MAPPING[snapshot_dict['state']]
        snapshot.label = snapshot_dict.get('marker')
        snapshot.created_at = timestamp(parse_datetime_iso(snapshot_dict['created_at']))
        snapshot.updated_at = timestamp(self.now)
        snapshot.order_amount = float(snapshot_dict['price'])
        snapshot.currency_code = OrderCurrencyCode(snapshot_dict['currency'])
        snapshot.origin = snapshot_dict.get('departure')
        snapshot.destination = snapshot_dict.get('arrival')
        snapshot.partner_id = self.partner_id
        snapshot.billing_order_id = self.billing_order_id

        LOG.debug(snapshot.as_dict())
        return snapshot
