# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date, datetime, timedelta

from dateutil.relativedelta import relativedelta
from lxml import etree
from six import text_type
from yt.wrapper import YtClient

from travel.avia.library.python.marker_helpers import AviaFlightRouteHelper
from travel.avia.library.python.references.partner import PartnerCache
from travel.avia.library.python.references.station import create_station_cache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, OnetwotripruAviaOrderSnapshot

LOG = get_logger(__name__)

SOURCE = PARTNER = 'onetwotripru'


class OnetwotripruCollector(HttpCollector):
    PARTNER_NAME = 'onetwotripru'
    BASE_URL = 'https://staging--bo.twiket.com/dashboard/getReportByAuth'
    REQUEST_TIMEOUT = 90
    STATUS_MAPPING = {
        'Issued': OrderStatus.CONFIRMED,
        'Void': OrderStatus.CANCELLED,
    }

    def __init__(self, options):
        LOG.info('OnetwotripruCollector INITs')
        super(OnetwotripruCollector, self).__init__(options)

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        self.auth = (options.username, options.password)

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(PARTNER)

        self.station = create_station_cache(yt_client)
        self.route_helper = AviaFlightRouteHelper(self.station)

        self.get_range_report = with_retries(
            func=self._get_range_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('-u', '--username', required=True)
        parser.add_argument('-p', '--password', required=True)

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-1)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

    def _get_snapshots(self):
        LOG.info('Getting snapshots for %r - %r', self.date_from, self.date_to)
        for snapshot in self.get_range_snapshots(self.date_from, self.date_to):
            if snapshot is None:
                continue
            yield snapshot

    def get_range_snapshots(self, date_from, date_to):
        """
        Собираем данные от партнера и раскладываем в snapshot
        """
        snapshot_date = date_from
        while snapshot_date <= date_to:
            r = self.get_range_report(snapshot_date, snapshot_date)
            for snapshot in self.parse_report(r.content):
                yield snapshot
            snapshot_date = snapshot_date + timedelta(days=1)

    def _get_range_report_once(self, date_from, date_to):
        params = {
            'from': date_from.strftime('%Y%m%d'),
            'to': date_to.strftime('%Y%m%d'),
        }
        LOG.info(
            'Getting data from %s date: %s - %s',
            self.base_url,
            date_from,
            date_to,
        )

        r = self.request_get(
            self.base_url,
            auth=self.auth,
            params=params,
            timeout=self.REQUEST_TIMEOUT,
        )
        return r

    def parse_report(self, content):
        try:
            tree = etree.fromstring(content, parser=etree.XMLParser(encoding='utf-8'))
        except etree.XMLSyntaxError:
            LOG.exception('Bad Onetwotripru XML')
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        for order_tree in tree.xpath('/Report/Repor'):
            try:
                status = self._parse_status(order_tree.attrib['status'])
                if not status:
                    LOG.info('Skipping report with unknown status: "%s"', order_tree.attrib['status'])
                    continue
                order = {
                    'status': status,
                    'label': text_type(order_tree.attrib['marker']),
                    'created_at': timestamp(self._convert_dt(order_tree.attrib['created_at'])),
                    'order_amount': float(order_tree.attrib['price']),
                    'currency_code': OrderCurrencyCode(order_tree.attrib['currency']),
                    'source': SOURCE,
                    'origin': text_type(order_tree.attrib['from']),
                    'destination': text_type(order_tree.attrib['to']),
                    'trip_type': self._parse_trip_type(order_tree.attrib['type']),
                    'partner_id': self.partner_id,
                    'billing_order_id': self.billing_order_id,
                }

                snapshot = OnetwotripruAviaOrderSnapshot.from_dict(
                    d=order,
                    convert_type=False,
                    ignore_unknown=True,
                )
                snapshot.update_partner_order_id(text_type(order_tree.attrib['orderid']))
                yield snapshot
            except Exception:
                LOG.exception('Onetwotripru parse error')
                raise

    def _parse_status(self, partner_status):
        return self.STATUS_MAPPING.get(partner_status)

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str, '%d.%m.%Y')

    @staticmethod
    def _parse_trip_type(trip_type):
        trip_type_map = {
            'OW': u'oneway',
            'RT': u'roundtrip',
            'OJ': u'openjaw',
            '': None
        }
        if trip_type not in trip_type_map:
            LOG.warning('Unknown trip_type %s', trip_type)
            return u'unknown'
        return trip_type_map[trip_type]
