# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date, datetime

from dateutil.relativedelta import relativedelta
from lxml import etree
from six import text_type
from yt.wrapper import YtClient

from travel.avia.library.python.marker_helpers import AviaFlightRouteHelper
from travel.avia.library.python.references.partner import PartnerCache
from travel.avia.library.python.references.station import create_station_cache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, OzonAviaOrderSnapshot

LOG = get_logger(__name__)

SOURCE = PARTNER = 'ozon'


class OzonCollector(HttpCollector):
    PARTNER_NAME = 'ozon'
    BASE_URL = 'https://www.ozon.travel/my/partners/period_xml.html'
    REQUEST_TIMEOUT = 90
    STATUS_MAPPING = {
        'InProgress': OrderStatus.PENDING,
        'Confirmed': OrderStatus.CONFIRMED,
        'Cancelled': OrderStatus.CANCELLED,
    }

    def __init__(self, options):
        super(OzonCollector, self).__init__(options)

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        self.auth = (options.username, options.password)

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(PARTNER)

        self.station = create_station_cache(yt_client)
        self.route_helper = AviaFlightRouteHelper(self.station)

        self.get_range_report = with_retries(
            func=self._get_range_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('-u', '--username', required=True)
        parser.add_argument('-p', '--password', required=True)

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-4)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

    def _get_snapshots(self):
        LOG.info('Getting snapshots for %r - %r', self.date_from, self.date_to)
        for snapshot in self.get_range_snapshots(self.date_from, self.date_to):
            if snapshot is None:
                continue
            yield snapshot

    def get_range_snapshots(self, date_from, date_to):
        """
        Собираем данные от партнера и раскладываем в snapshot
        """
        r = self.get_range_report(date_from, date_to)
        for snapshot in self.parse_report(r.content):
            yield snapshot

    def _get_range_report_once(self, date_from, date_to):
        params = {
            'DateStart': date_from.strftime('%Y-%m-%d'),
            'Date': date_to.strftime('%Y-%m-%d'),
        }
        LOG.info(
            'Getting data from %s date: %s - %s',
            self.base_url,
            date_from,
            date_to,
        )
        r = self.request_get(
            self.base_url,
            auth=self.auth,
            params=params,
            timeout=self.REQUEST_TIMEOUT,
        )
        return r

    def parse_report(self, content):
        try:
            tree = etree.fromstring(content, parser=etree.XMLParser(encoding='utf-8'))
        except etree.XMLSyntaxError:
            LOG.exception('Bad Ozon XML')
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        for order_tree in tree.xpath('//Report/Orders/Order'):
            try:
                if order_tree.get('OrderType') != 'AIR':
                    continue
                order_dict = {row.tag: row for row in order_tree}
                flight_dicts = [
                    {
                        'from': text_type(flight.get('From')),
                        'to': text_type(flight.get('To')),
                        'departure_dt': self._convert_dt(flight.get('Date')),
                        'arrival_dt': self._convert_dt(flight.get('Date')),
                    }
                    for flight in order_dict['FlightRoute']
                ]
                order = {
                    'status': self._parse_status(order_tree.get('Status')),
                    'label': text_type(order_tree.get('PartnerRef')),
                    'created_at': timestamp(self._convert_dt(order_tree.get('CreateDate'))),
                    'order_amount': float(order_tree.get('Sum')),
                    'currency_code': OrderCurrencyCode.RUB,
                    'flights': flight_dicts,
                    'source': SOURCE,
                    'partner_id': self.partner_id,
                    'billing_order_id': self.billing_order_id,
                }
                self.route_helper.fillin_trip_info(order)
                snapshot = OzonAviaOrderSnapshot.from_dict(
                    d=order,
                    convert_type=False,
                    ignore_unknown=True,
                )
                snapshot.update_partner_order_id(text_type(order_tree.get('Number')))
                yield snapshot
            except Exception:
                LOG.exception('Ozon parse error')

    def _parse_status(self, partner_status):
        return self.STATUS_MAPPING[partner_status]

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str.split('.')[0], '%Y-%m-%dT%H:%M:%S')
