# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import string
import tempfile
import traceback
from collections import namedtuple
from contextlib import contextmanager
from datetime import date, datetime

import openpyxl
import paramiko
from dateutil.relativedelta import relativedelta
from yt.wrapper import YtClient

from travel.avia.library.python.references.partner import PartnerCache
from travel.cpa.collectors.lib.collector import Collector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.lib_datetime import parse_datetime_iso
from travel.cpa.lib.lib_datetime import timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderStatus, PobedaAviaOrderSnapshot

LOG = get_logger(__name__)

SOURCE = PARTNER = 'pobeda'


class PobedaCollector(Collector):
    PARTNER_NAME = 'pobeda'
    BASE_URL = 'ftp.pobeda.aero'
    PORT = 22
    DATA_FILE_TEMPLATE = 'POBEDA-YANDEX-{date}.xlsx'

    SSH_TIMEOUT = 120
    STATUS_MAPPING = {
        u'Отмена': OrderStatus.CANCELLED
    }

    def __init__(self, options):
        super(PobedaCollector, self).__init__()

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        self.test_only_local_path = options.test_only_local_path
        self._ftp_host = options.base_url
        self._login = options.username
        self._password = options.password
        self.port = options.port
        self.timeout = options.timeout

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(PARTNER)

        self.get_day_report = with_retries(
            func=self._get_day_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)
        parser.add_argument('--port', type=int, default=cls.PORT)
        parser.add_argument('-u', '--username', required=True)
        parser.add_argument('--password', required=True)
        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-4)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())
        parser.add_argument('--test-only-local-path')
        parser.add_argument('--timeout', type=int, default=cls.SSH_TIMEOUT)

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

    def _get_snapshots(self):
        for filename in self._get_remote_file_names():
            LOG.info('Getting file  %r', filename)
            for snapshot in self.get_day_snapshots(filename):
                if snapshot is None:
                    continue
                yield snapshot

    def get_day_snapshots(self, filename):
        """
        Собираем данные от партнера и раскладываем в snapshot
        """
        r = self.get_day_report(filename)
        for snapshot in r:
            yield snapshot

    @contextmanager
    def _get_mock_workbook(self, filename):
        yield openpyxl.load_workbook(filename, read_only=True)

    def _get_remote_file_names(self):
        if self.test_only_local_path:
            yield self.test_only_local_path
            return
        with paramiko.SSHClient() as ssh_client:
            ssh_client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
            try:
                ssh_client.connect(
                    hostname=self._ftp_host,
                    username=self._login,
                    password=self._password,
                    port=self.port,
                    look_for_keys=False,
                    allow_agent=False,
                    timeout=self.timeout,
                )

                with ssh_client.open_sftp() as sftp:
                    files = sftp.listdir()

            except Exception:
                LOG.exception('Can not receive list of remote files')
                raise
        first_file = self.DATA_FILE_TEMPLATE.format(date=self.date_from.strftime('%Y-%m-%d'))
        last_file = self.DATA_FILE_TEMPLATE.format(date=self.date_to.strftime('%Y-%m-%d'))
        for file in files:
            if first_file <= file <= last_file:
                yield file

    @contextmanager
    def _get_day_workbook(self, filename):
        with paramiko.SSHClient() as ssh_client:
            ssh_client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
            try:
                ssh_client.connect(
                    hostname=self._ftp_host,
                    username=self._login,
                    password=self._password,
                    port=self.port,
                    look_for_keys=False,
                    allow_agent=False,
                    timeout=self.timeout,
                )

                with ssh_client.open_sftp() as sftp:
                    # Define the file that you want to download from the remote directory
                    try:
                        with tempfile.NamedTemporaryFile(suffix=".xlsx") as temp:
                            sftp.get(filename, temp.name)
                            yield openpyxl.load_workbook(filename=temp.name, read_only=True)

                    except IOError:
                        LOG.warn('No such remote file: %s', filename)

            except Exception:
                LOG.exception(traceback.format_exc())
                raise

    def _get_day_report_once(self, filename):

        if self.test_only_local_path:
            get_day_workbook = self._get_mock_workbook
        else:
            get_day_workbook = self._get_day_workbook

        with get_day_workbook(filename) as wb:
            CellProcessor = namedtuple('CellProcessor', ['destination', 'process'])
            # List the resulting dictionary keys to output data and the functions to get that data
            cell_processors = {
                u'Время создания': CellProcessor('created_at', self._parse_date),
                u'PNR': CellProcessor('order_id', self._as_is),
                u'Статус брони': CellProcessor(
                    'status', lambda value: self.STATUS_MAPPING.get(value, OrderStatus.CONFIRMED)),
                u'Маршрут': CellProcessor('trip_type', self._trip_type),
                u'marker': CellProcessor('label', self._as_is),
                u'Тариф': CellProcessor('_tariff', self._to_float),
                u'Возврат тарифа': CellProcessor('_tariff_refund', self._to_float),
                u'Услуги': CellProcessor('_services', self._to_float),
                u'Возврат услуг': CellProcessor('_services_refund', self._to_float),
                u'Комиссия,руб': CellProcessor('_commission', self._to_float),
            }

            headers = []
            for sheet_name in wb.get_sheet_names():
                sheet = wb[sheet_name]
                for index, row in enumerate(sheet.rows):
                    # Headers are in the second row, so skip the first one
                    if all(cell.value is None for cell in row):
                        continue

                    if not bool(headers):
                        headers = [string.strip(cell.value) if cell.value is not None else None
                                   for cell in row]
                        continue

                    order = {}
                    for col_index, cell in enumerate(row):
                        if cell.value is not None:
                            processor = cell_processors.get(headers[col_index])
                            if processor:
                                result = processor.process(cell.value)
                                if result is not None:
                                    order[processor.destination] = result

                    if order.get('created_at') is None or order.get('label') is None:
                        continue

                    price = max(
                        0.,
                        self._net_value(order, '_tariff', '_tariff_refund') +
                        self._net_value(order, '_services', '_services_refund')
                    )
                    order['order_amount'] = self._round(price)
                    order['partner_id'] = self.partner_id
                    order['billing_order_id'] = self.billing_order_id
                    if price == 0:
                        order['status'] = OrderStatus.CANCELLED

                    snapshot = PobedaAviaOrderSnapshot.from_dict(d=order, ignore_unknown=True)
                    snapshot.update_partner_order_id(order['order_id'])
                    yield snapshot

    @staticmethod
    def _round(value):
        return int((value * 100) + 0.5) / 100.0

    @staticmethod
    def _as_is(value):
        return value

    @staticmethod
    def _parse_date(value):
        return timestamp(datetime.strptime(value, '%d.%m.%Y %H:%M')) if value else None

    @staticmethod
    def _trip_type(value):
        return {
            'OW': u'oneway',
            'RT': u'roundtrip',
        }.get(value)

    @staticmethod
    def _net_value(dict, positive_key, negative_key):
        return PobedaCollector._pop_float(dict, positive_key) - PobedaCollector._pop_float(dict, negative_key)

    @staticmethod
    def _pop_float(dict, key):
        return PobedaCollector._to_float(dict.pop(key, 0.))

    @staticmethod
    def _to_float(value):
        try:
            return float(value)
        except:
            return 0.

    def parse_report(self, content):
        return content
