# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from collections import namedtuple
from datetime import date, datetime

import six
from dateutil.relativedelta import relativedelta
from requests.auth import HTTPBasicAuth
from yt.wrapper import YtClient

from travel.avia.library.python.references.partner import PartnerCache

from travel.cpa.lib.errors import ProcessError, ErrorType
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, SSevenAviaOrderSnapshot

LOG = get_logger(__name__)


OrderPriceInfo = namedtuple('OrderPriceInfo', (
    'order_amount',  # Сумма заказа
    'service_fee',  # Сумма, за которую комиссию мы не получаем, входит в order_amount
    'partner_profit',  # Сумма нашей выручки по расчетам партнера
    'currency',
))

ZERO_PRICE_INFO = OrderPriceInfo(
    order_amount=0.,
    service_fee=0.,
    partner_profit=0.,
    currency='RUB',
)


class SSevenCollector(HttpCollector):
    PARTNER_NAME = 's_seven'
    BASE_URL = 'https://msestat.api.s7.ru/api/statistics'
    REQUEST_TIMEOUT = 90
    STATUS_MAPPING = {
        'BOOKED': OrderStatus.PENDING,
        'COMPLETED': OrderStatus.CONFIRMED,
        'HISTORIC': OrderStatus.CONFIRMED,
        'CANCELLED': OrderStatus.CANCELLED,
    }

    # еще может быть MULTI_TRIP - но это любой составной маршрут, может быть как ROUND_TRIP так и ONE_WAY
    # могут быть только эти 3 зтипа маршрута, поэтому отсутствие типа == MULTI_TRIP
    TRIP_TYPE_MAPPING = {
        'ROUND_TRIP': 'roundtrip',
        'ONE_WAY': 'oneway',
    }

    def __init__(self, options):
        super(SSevenCollector, self).__init__(options)

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()
        if options.valid_from:
            self.date_from = max(self.date_from, parse_datetime_iso(options.valid_from).date())

        self.auth = HTTPBasicAuth(options.username, options.password)

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(self.PARTNER_NAME)

        self.get_day_report = with_retries(
            func=self._get_day_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('-u', '--username', default='Yandex')
        parser.add_argument('-p', '--password', required=True)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-4)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())
        parser.add_argument('--valid-from', default=None)

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

    def _get_snapshots(self):
        for day in iter_day(self.date_from, self.date_to):
            LOG.info('Getting snapshots for %r', day)
            for snapshot in self.get_day_snapshots(day):
                yield snapshot

    def get_day_snapshots(self, day):
        for raw_order in self.get_day_report(day):
            snapshot = self.parse_s_seven_order(raw_order)
            if snapshot:
                yield snapshot

    def _get_day_report_once(self, day):
        params = {
            'date1': day.strftime('%Y-%m-%d'),
            'date2': day.strftime('%Y-%m-%d'),
        }
        response = self.request_get(
            self.base_url,
            auth=self.auth,
            params=params,
            timeout=self.REQUEST_TIMEOUT,
        )
        try:
            data = response.json()
        except ValueError:
            LOG.exception('Got invalid json from s_seven')
            LOG.debug(response.text)
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        return data.get('orders', [])

    def parse_s_seven_order(self, raw_order):
        order_price_info = self.calc_price_info(raw_order['payments'], raw_order['number'])

        # airs почти всегда один, из него мы берем trip_type и направление, поэтому подойдет первый.
        # В наших данных "на сейчас" air всегда однин, но коллеги из S7 утверждают, что может быть больше.
        air = raw_order['airs'][0]
        # Даже если есть round trip, то берем пункты из маршрута туда
        # Иногда список routes пустой, бывает даже для статуса COMPLETED
        direction = air['routes'][0] if air['routes'] else None
        order = {
            'status': self._parse_status(raw_order['status']),
            'created_at': timestamp(self._parse_dt(raw_order['created'])),
            'order_amount': order_price_info.order_amount,
            'service_fee': order_price_info.service_fee,
            'partner_profit': order_price_info.partner_profit,
            'currency_code': OrderCurrencyCode(order_price_info.currency),
            'label': raw_order.get('partner_marker'),
            'origin': direction and direction['origin'],
            'destination': direction and direction['destination'],
            'trip_type': self._parse_trip_type(air['tripType']),
            'partner_id': self.partner_id,
            'billing_order_id': self.billing_order_id,
        }

        snapshot = SSevenAviaOrderSnapshot.from_dict(
            d=order,
            convert_type=False,
            ignore_unknown=True,
        )
        snapshot.update_partner_order_id(six.text_type(raw_order['number']))
        return snapshot

    def calc_price_info(self, payments, order_id):
        if not payments:
            LOG.warning('Empty payments detected %s', order_id)
            return ZERO_PRICE_INFO

        order_amount = 0.
        order_service_fee = 0.
        order_partner_profit = 0.
        currency = payments[0]['pricing']['total']['salePrice']['currency']
        for payment in payments:
            total = payment['pricing']['total']
            payment_currency = total['salePrice']['currency']
            self._check_currency(payment_currency, currency)
            order_amount += total['salePrice']['amount']

            service_fee = total.get('serviceFee')
            if service_fee and service_fee['amount']:
                self._check_currency(service_fee['currency'], currency)
                order_service_fee += service_fee['amount']

            partner_profit = payment['pricing'].get('partnerFee', {}).get('salePrice')
            if partner_profit and partner_profit['amount']:
                self._check_currency(partner_profit['currency'], currency)
                order_partner_profit += partner_profit['amount']

        return OrderPriceInfo(
            order_amount=order_amount,
            service_fee=order_service_fee,
            partner_profit=order_partner_profit,
            currency=currency,
        )

    def _check_currency(self, actual, expected):
        if actual != expected:
            LOG.error('Different currencies "%s" and "%s" in one order', expected, actual)
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

    def _parse_status(self, partner_status):
        return self.STATUS_MAPPING[partner_status.upper()]

    def _parse_trip_type(self, trip_type):
        return self.TRIP_TYPE_MAPPING.get(trip_type)

    @staticmethod
    def _parse_dt(dt_str):
        return datetime.strptime(dt_str[:19], '%Y-%m-%dT%H:%M:%S')
