# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date, datetime

from dateutil.relativedelta import relativedelta
from lxml import etree
from six import text_type
from yt.wrapper import YtClient

from travel.avia.library.python.marker_helpers import AviaFlightRouteHelper
from travel.avia.library.python.references.partner import PartnerCache
from travel.avia.library.python.references.station import create_station_cache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib import order_snapshot
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger

LOG = get_logger(__name__)


class SmartaviaCollector(HttpCollector):
    REQUEST_TIMEOUT = 90
    PARTNER_NAME = 'smartavia'
    BASE_URL = 'https://bo-prod-api.flyways.ru/api/meta/report/5n'
    STATUS_MAPPING = {
        'PROCESSING': order_snapshot.OrderStatus.PENDING,
        'PAID': order_snapshot.OrderStatus.CONFIRMED,
        'CANCELED': order_snapshot.OrderStatus.CANCELLED,
    }

    def __init__(self, options):
        super(SmartaviaCollector, self).__init__(options)
        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()
        self.password = options.password

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(self.PARTNER_NAME)

        self.station = create_station_cache(yt_client)
        self.route_helper = AviaFlightRouteHelper(self.station)

        self.get_day_report = with_retries(
            func=self._get_day_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('-p', '--password', required=True)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-4)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

    def _get_snapshots(self):
        for day_date in iter_day(self.date_from, self.date_to):
            LOG.info('Getting snapshots for %r', day_date)
            for snapshot in self.get_day_snapshots(day_date):
                if snapshot is None:
                    continue
                yield snapshot

    def get_day_snapshots(self, day_date):
        """
        Собираем данные от партнера и раскладываем в snapshot
        """
        r = self.get_day_report(day_date)
        for snapshot in self.parse_report(r.content):
            yield snapshot

    def _get_day_report_once(self, day_date):
        params = {
            'date1': day_date.strftime('%Y-%m-%d'),
            'date2': day_date.strftime('%Y-%m-%d'),
            'password': self.password,
        }
        LOG.info(
            'Getting data from %s date range: %s - %s',
            self.base_url,
            params['date1'],
            params['date2'],
        )
        r = self.request_get(
            self.base_url,
            params=params,
            timeout=self.REQUEST_TIMEOUT,
        )
        return r

    def parse_report(self, content):
        try:
            tree = etree.fromstring(content)
        except etree.XMLSyntaxError:
            LOG.exception('Bad Smartavia XML. response start with "%s"', content[:40])
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        for order_tree in tree.xpath('//bookings/booking'):
            order_dict = {row.tag: row.text for row in order_tree}
            flight_dicts = []
            try:
                flight_dicts = [
                    {
                        'from': flight['departure'],
                        'to': flight['arrival'],
                        'departure_dt': self._convert_flight_dt(
                            flight['departureDate'] + ' ' + flight['departureTime']
                        ),
                        'arrival_dt': self._convert_flight_dt(
                            flight['arrivalDate'] + ' ' + flight['arrivalTime']
                        ),
                    }
                    for flight in
                    [
                        {
                            text_type(param.tag): text_type(param.text) for param in flight_tree
                        }
                        for flight_tree in order_tree.xpath('segment/flight')
                    ]
                ]
            except ValueError:
                LOG.info('Unable to parse flight data, ignoring')

            order = {
                'status': self._parse_status(order_dict.get('state')),
                'label': text_type(order_dict.get('marker')),
                'created_at': timestamp(self._convert_dt(order_dict.get('created_at'))),
                'updated_at': timestamp(self.now),
                'order_amount': float(order_dict.get('price')),
                'currency_code': order_snapshot.OrderCurrencyCode(order_dict.get('currency')),
                'partner_profit': float(order_dict.get('profit')),
                'flights': flight_dicts,
                'partner_id': self.partner_id,
                'billing_order_id': self.billing_order_id,
            }
            if flight_dicts:
                self.route_helper.fillin_airports(order)
            snapshot = order_snapshot.SmartaviaAviaOrderSnapshot.from_dict(
                d=order,
                convert_type=False,
                ignore_unknown=True,
            )
            order_id = order_dict.get('id')
            snapshot.update_partner_order_id(text_type(order_id))
            LOG.debug('%r', snapshot.as_dict())
            yield snapshot

    def _parse_status(self, partner_status):
        return self.STATUS_MAPPING[partner_status]

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str.split('.')[0], '%Y-%m-%d %H:%M:%S')

    @staticmethod
    def _convert_flight_dt(dt_str):
        possible_formats = [
            '%Y-%m-%d %H:%M:%S.%f',
            '%Y-%m-%d %H:%M:%S',
            '%Y-%m-%d %H:%M.%f',
            '%Y-%m-%d %H:%M',
        ]
        errors = []
        for layout in possible_formats:
            try:
                return datetime.strptime(dt_str, layout)
            except ValueError as e:
                errors.append(e)
                continue
        raise ValueError('Cannot parse time with any of layouts: {}'.format(errors))
