# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date, datetime

from dateutil.relativedelta import relativedelta
from six import text_type
from yt.wrapper import YtClient

from travel.avia.library.python.references.partner import PartnerCache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, SvyaznoyAviaOrderSnapshot

LOG = get_logger(__name__)

SOURCE = PARTNER = 'svyaznoy'


class SvyaznoyCollector(HttpCollector):
    PARTNER_NAME = 'svyaznoy'
    BASE_URL = 'http://meta.svyaznoy.travel/multireq/api/stat'
    REQUEST_TIMEOUT = 90
    STATUS_MAPPING = {
        'notpaid': OrderStatus.PENDING,
        'issued': OrderStatus.CONFIRMED,
        'refund': OrderStatus.REFUNDED,
        'cancelled': OrderStatus.CANCELLED,
        'exchange': OrderStatus.CONFIRMED,
    }

    def __init__(self, options):
        super(SvyaznoyCollector, self).__init__(options)

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        self.auth = (options.username, options.password)

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(PARTNER)

        self.get_day_report = with_retries(
            func=self._get_day_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('-u', '--username', required=True)
        parser.add_argument('-p', '--password', required=True)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-4)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

    def _get_snapshots(self):
        for day_date in iter_day(self.date_from, self.date_to):
            LOG.info('Getting snapshots for %r', day_date)
            for snapshot in self.get_day_snapshots(day_date):
                if snapshot is None:
                    continue
                yield snapshot

    def get_day_snapshots(self, day_date):
        """
        Собираем данные от партнера и раскладываем в snapshot
        """
        order_list = self.get_day_report(day_date)
        for snapshot in self.parse_report(order_list):
            yield snapshot

    def _get_day_report_once(self, day_date):
        (username, password) = self.auth
        params = {
            'date_start': day_date.strftime('%Y-%m-%d'),
            'date_end': day_date.strftime('%Y-%m-%d'),
            'source': username,
            'passwd': password,
            'gmt': '0',
        }
        LOG.info(
            'Getting data from %s date range: %s - %s',
            self.base_url,
            params['date_start'],
            params['date_end'],
        )
        r = self.request_get(
            self.base_url,
            params=params,
            timeout=self.REQUEST_TIMEOUT,
        )
        LOG.debug(r.text)
        try:
            order_list = r.json()
        except ValueError:
            LOG.exception('Bad SVYAZNOY JSON')
            raise ProcessError(ErrorType.ET_PARTNER_DATA)
        return order_list

    def parse_report(self, order_list):
        for order_dict in order_list:
            (orig, dest) = order_dict.get('direction').split('/')
            status = order_dict.get('status')
            status = status.lower() if isinstance(status, str) else status
            order = {
                'status': self._parse_status(status),
                'label': order_dict.get('marker'),
                'created_at': timestamp(self._convert_dt(order_dict.get('booking_date'))),
                'order_amount': float(order_dict.get('price')),
                'currency_code': OrderCurrencyCode.RUB,
                'origin': orig,
                'destination': dest,
                'trip_type': 'unknown',
                'source': SOURCE,
                'partner_id': self.partner_id,
                'billing_order_id': self.billing_order_id,
            }
            snapshot = SvyaznoyAviaOrderSnapshot.from_dict(
                d=order,
                convert_type=False,
                ignore_unknown=True,
            )
            snapshot.update_partner_order_id(text_type(order_dict.get('order_id')))
            yield snapshot

    @staticmethod
    def _parse_price(input_str):
        """
        Разбиваем строчку с ценой на цену и валюту
        :param basestring input_str: строка вида 1123.01 RUR
        :return: кортеж вида (float_price, currency)
        """
        price_str, currency_str = input_str.split(' ')
        price = float(price_str)
        if currency_str == 'RUR':
            currency_str = 'RUB'
        currency = OrderCurrencyCode(currency_str)
        return price, currency

    def _parse_status(self, partner_status):
        return self.STATUS_MAPPING[partner_status.lower()]

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str, '%Y-%m-%d %H:%M:%S')
