# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date, datetime

from six import text_type
from yt.wrapper import YtClient

from dateutil.relativedelta import relativedelta
from travel.avia.library.python.references.partner import PartnerCache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, TinkoffAviaOrderSnapshot

LOG = get_logger(__name__)


class TinkoffCollector(HttpCollector):
    SOURCE = PARTNER_NAME = 'tinkoff1'
    BASE_URL = 'https://api-travel.tinkoff.ru/api/v1/metasearch/yandex/statistics'
    REQUEST_TIMEOUT = 90
    STATUS_MAPPING = {
        'processing': OrderStatus.PENDING,
        'paid': OrderStatus.CONFIRMED,
        'cancelled': OrderStatus.CANCELLED,
    }

    def __init__(self, options):
        super(TinkoffCollector, self).__init__(options)
        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(self.PARTNER_NAME)

        self.get_day_report = with_retries(
            self.get_day_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-4)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

    def _get_snapshots(self):
        for day_date in iter_day(self.date_from, self.date_to):
            LOG.info('Getting snapshots for %r', day_date)
            for snapshot in self.get_day_snapshots(day_date):
                if snapshot is None:
                    continue
                yield snapshot

    def get_day_snapshots(self, day_date):
        report = self.get_day_report(day_date)
        for raw_snapshot in report.splitlines()[1:]:
            yield self.get_order_snapshot(raw_snapshot)

    def get_day_report_once(self, day_date):
        params = {
            'date_from': day_date.strftime('%Y-%m-%d'),
            'date_to': day_date.strftime('%Y-%m-%d'),
        }
        rsp = self.request_get(self.base_url, params=params, timeout=self.REQUEST_TIMEOUT)
        return text_type(rsp.content)

    def get_order_snapshot(self, raw_snapshot):
        origin, destination, created_at, price, currency, adults, children, infants, marker, order_id, status = \
            raw_snapshot.split(',')

        snapshot = TinkoffAviaOrderSnapshot()
        snapshot.update_partner_order_id(order_id)
        snapshot.status = self.STATUS_MAPPING[status]
        snapshot.label = marker
        snapshot.created_at = timestamp(self._convert_dt(created_at))
        snapshot.order_amount = float(price)
        snapshot.currency_code = OrderCurrencyCode(currency)
        snapshot.partner_id = self.partner_id
        snapshot.billing_order_id = self.billing_order_id
        snapshot.origin = origin
        snapshot.destination = destination
        snapshot.adults = int(adults)
        snapshot.children = int(children)
        snapshot.infants = int(infants)
        return snapshot

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str, '%Y-%m-%d')
