# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date, datetime
from yt.wrapper import YtClient

from dateutil.relativedelta import relativedelta
from travel.avia.library.python.marker_helpers import AviaFlightRouteHelper
from travel.avia.library.python.references.partner import PartnerCache
from travel.avia.library.python.references.station import create_station_cache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ProcessError, ErrorType
from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, TutuAviaOrderSnapshot

LOG = get_logger(__name__)


class TutuCollector(HttpCollector):
    PARTNER_NAME = 'tutu'
    PARTNER_CODE = 'tutu'
    REQUEST_TIMEOUT = 90
    BASE_URL = 'https://avia.tutu.ru/yapi/orders/'
    SNAPSHOT_CLS = TutuAviaOrderSnapshot
    STATUS_MAPPING = {
        'booked': OrderStatus.PENDING,
        'paid': OrderStatus.CONFIRMED,
        'cancel': OrderStatus.CANCELLED,
        'canceled': OrderStatus.CANCELLED,
        'refunded': OrderStatus.REFUNDED,
    }

    def __init__(self, options):
        super(TutuCollector, self).__init__(options)
        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.station = create_station_cache(yt_client)
        self.route_helper = AviaFlightRouteHelper(self.station)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(self.PARTNER_NAME)

        self.get_day_report = with_retries(
            self.get_day_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-1)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

        parser.add_argument('--base-url', default=cls.BASE_URL)

    def _get_snapshots(self):
        for day_date in iter_day(self.date_from, self.date_to):
            LOG.info('Getting snapshots for %r', day_date)
            for snapshot in self.get_day_snapshots(day_date):
                if snapshot is None:
                    continue
                yield snapshot

    def get_day_report_once(self, day_date):
        params = {
            'dateFrom': day_date.strftime('%Y-%m-%d'),
            'dateTo': (day_date + relativedelta(days=1)).strftime('%Y-%m-%d'),
        }
        rsp = self.request_get(self.base_url, params=params, timeout=self.REQUEST_TIMEOUT)
        return rsp.content

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str, '%Y-%m-%d')

    def get_day_snapshots(self, day_date):
        import ujson

        orders = []
        response = {}
        try:
            report = self.get_day_report(day_date)
            response = ujson.loads(report)
            bookings = response['bookings']['booking']
        except ValueError:
            LOG.exception('Bad Tutu JSON')
            LOG.debug(response)
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        try:
            for booking in bookings:
                orders.append(self.get_order_snapshot(booking))
            return orders
        except ValueError:
            LOG.exception('Bad %s DATA %s', self.PARTNER_CODE.upper(), report[:1000])
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

    def get_order_snapshot(self, order_dict):
        try:
            flights = map(self.get_mapped_flight, order_dict['segment']['flight'])
            flights = sorted(flights, key=lambda flight: flight['departure_dt'])
            route_info = self.route_helper.fillin_trip_info({'flights': flights})
            snapshot = self.SNAPSHOT_CLS.from_dict(route_info, ignore_unknown=True)
        except:
            LOG.exception('Could not get route information')
            snapshot = self.SNAPSHOT_CLS()

        snapshot.update_partner_order_id(order_dict.get('id'))
        snapshot.status = self.STATUS_MAPPING[order_dict.get('status')]
        snapshot.label = order_dict.get('marker')
        snapshot.created_at = timestamp(datetime.strptime(order_dict.get('created_at'), '%Y-%m-%d %H:%M:%S'))
        snapshot.updated_at = timestamp(self.now)
        snapshot.order_amount = float(order_dict.get('price'))
        snapshot.currency_code = OrderCurrencyCode(order_dict.get('currency'))
        snapshot.adults = order_dict.get('adults')
        snapshot.children = order_dict.get('children')
        snapshot.infants = order_dict.get('infants')
        snapshot.partner_id = self.partner_id
        snapshot.billing_order_id = self.billing_order_id
        snapshot.unisearchquery = order_dict.get('unisearchquery')

        LOG.debug(snapshot.as_dict())
        return snapshot

    @staticmethod
    def get_mapped_flight(flight):
        return {
            'from': flight.get('departure'),
            'to': flight.get('arrival'),
            'departure_dt': TutuCollector._convert_dt(flight.get('departureDate')),
            'arrival_dt': TutuCollector._convert_dt(flight.get('arrivalDate')),
        }
