# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date, datetime, timedelta
from hashlib import md5

from dateutil.relativedelta import relativedelta
from six import text_type
from yt.wrapper import YtClient

from travel.avia.library.python.marker_helpers import AviaFlightRouteHelper
from travel.avia.library.python.references.partner import PartnerCache
from travel.avia.library.python.references.station import create_station_cache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, UtairAviaOrderSnapshot

LOG = get_logger(__name__)


class UtairCollector(HttpCollector):
    PARTNER_NAME = 'utair'
    PARTNER_CODE = 'utair'
    BASE_URL = 'https://tais-api.utair.ru/bitrix/components/travelshop/ibe.rest/GetOrderList/'
    REQUEST_TIMEOUT = 60
    STATUS_MAPPING = {
        'S': OrderStatus.PENDING,
        'B': OrderStatus.PENDING,
        'X': OrderStatus.CANCELLED,
        'T': OrderStatus.CONFIRMED,
    }
    TRIP_TYPE_MAPPING = {
        'RT': 'roundtrip',
        'OW': 'oneway',
        'OJ': 'openjaw',
    }

    def __init__(self, options):
        super(UtairCollector, self).__init__(options)

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        self._login = options.login
        self._password = options.password
        self._shared_secret = options.shared_secret

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(self.PARTNER_CODE)

        self.station = create_station_cache(yt_client)
        self.route_helper = AviaFlightRouteHelper(self.station)

        self.get_day_report = with_retries(
            func=self._get_day_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('-u', '--login', required=True)
        parser.add_argument('-p', '--password', required=True)
        parser.add_argument('-s', '--shared-secret', required=True)

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-4)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

    def _get_snapshots(self):
        for day_date in iter_day(self.date_from, self.date_to):
            LOG.info('Getting snapshots for %r', day_date)
            for snapshot in self.get_day_snapshots(day_date):
                if snapshot is None:
                    continue
                yield snapshot

    def get_day_snapshots(self, day_date):
        """
        Собираем данные от партнера и раскладываем в snapshot
        """
        response = self.get_day_report(day_date)
        for snapshot in self.parse_report(response):
            yield snapshot

    @staticmethod
    def _ensure_response(data):
        if 'error' not in data:
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        error = data['error']
        if error['code'] != 'OK':
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

    def _get_request_body(self, date):
        from_date = date
        to_date = date + timedelta(days=1)

        from_date = from_date.strftime('%d.%m.%Y')
        to_date = to_date.strftime('%d.%m.%Y')

        LOG.info('Requesting for date range %s - %s', from_date, to_date)

        session_token = ':'.join([self._login, self._password, 'N'])
        hash = self._get_hash(session_token, from_date, to_date)

        return dict(
            session_token=session_token,
            timestamp_from=from_date,
            timestamp_to=to_date,
            hash=hash,
        )

    def _get_hash(self, session_token, from_date, to_date):
        m = md5()
        m.update((session_token + from_date + to_date + self._shared_secret).encode('utf-8'))
        return m.hexdigest()

    def _get_day_report_once(self, day_date):
        LOG.info(
            'Getting data from %s date: %s',
            self.base_url,
            day_date,
        )
        request_body = self._get_request_body(day_date)
        return self.request_post(self.base_url, json=request_body, timeout=self.REQUEST_TIMEOUT).json()

    def parse_report(self, data):
        self._ensure_response(data)

        if not data['orders']:
            LOG.info('Order list is empty')
            return

        currency = OrderCurrencyCode(data['currency'])
        for order_dict in data['orders']:
            try:
                flights = [
                    self._parse_flight(flight_dict)
                    for flight_dict
                    in order_dict['segments']
                ]

                order = {
                    'created_at': timestamp(self._convert_dt(order_dict.get('booking_timestamp'))),
                    'order_amount': float(order_dict.get('full_price')),
                    'currency_code': currency,
                    'status': self._parse_status(order_dict.get('status')),
                    'label': order_dict.get('partner_data'),
                    'flights': flights,
                    'origin': order_dict['departure_point'],
                    'destination': order_dict['arrival_point'],
                    'trip_type': self._get_trip_type(order_dict['owrt']),
                    'partner_id': self.partner_id,
                    'billing_order_id': self.billing_order_id,
                }
                snapshot = UtairAviaOrderSnapshot.from_dict(
                    d=order,
                    convert_type=False,
                    ignore_unknown=True,
                )
                snapshot.update_partner_order_id(text_type(order_dict.get('order_id')))
                yield snapshot
            except Exception:
                LOG.exception('Utair parse error')
                raise

    def _parse_flight(self, segment):
        try:
            return {
                'from': segment['departure_airport_code'],
                'to': segment['arrival_airport_code'],
                'departure_dt': self._convert_dt(segment['departure_utc']),
                'arrival_dt': self._convert_dt(segment['arrival_utc']),
            }
        except Exception:
            LOG.exception(
                'Error parse flights for segment %s',
                repr(segment)
            )
            raise

    def _parse_status(self, partner_status):
        return self.STATUS_MAPPING[partner_status]

    @staticmethod
    def _convert_dt(dt_str):
        if not dt_str:
            return dt_str
        return datetime.strptime(dt_str, '%d.%m.%Y %H:%M:%S')

    def _get_trip_type(self, code):
        if code in self.TRIP_TYPE_MAPPING:
            return self.TRIP_TYPE_MAPPING[code]
        LOG.warn('Unexpected trip type %s', code)
