# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date

from dateutil.relativedelta import relativedelta
from six import text_type
import requests

from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.lib_datetime import iter_day_reversed, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import BookingOrderSnapshot, OrderCurrencyCode, OrderStatus


LOG = get_logger(__name__)


class BookingCollector(HttpCollector):
    PARTNER_NAME = 'booking'
    SET_HOST_HEADER = True

    BASE_URL = 'https://secure-distribution-xml.booking.com/2.5/json/bookingDetails'
    PAGE_SIZE = 1000
    REQUEST_TIMEOUT = 90

    STATUS_MAPPING = {
        'booked': OrderStatus.PENDING,
        'cancelled': OrderStatus.REFUNDED,
        'stayed': OrderStatus.CONFIRMED,
    }

    def __init__(self, options):
        super(BookingCollector, self).__init__(options)
        self.options = options
        self.client = requests.session()
        self.client.get = self.metered_get_with_retries(
            self.client.get, counter=self.metrics, key='collector.events.http_retry'
        )
        self.get_result = with_retries(
            self.get_result_once,
            max_attempts=5,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )
        self.client.auth = self.options.username, self.options.password

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        self.updates_since = None
        updates_last_days = self.options.updates_last_days
        if updates_last_days is not None:
            self.updates_since = date.today() - relativedelta(days=updates_last_days)

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)
        parser.add_argument('-u', '--username', required=True)
        parser.add_argument('-p', '--password', required=True)
        parser.add_argument('--date-from', default=(date.today() - relativedelta(months=3)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())
        parser.add_argument('--updates-last-days', type=int)

    def _get_snapshots(self):
        updates_since = self.updates_since
        if updates_since is not None:
            LOG.info('Getting order updates only')
            for snapshot in self.get_update_snapshots(updates_since):
                yield snapshot
            return
        LOG.info('Getting orders by creation date')
        for day_date in iter_day_reversed(self.date_to, self.date_from):
            for snapshot in self.get_day_snapshots(day_date):
                yield snapshot

    def get_day_snapshots(self, day_date):
        LOG.info('Processing {}'.format(day_date))
        for row in self.get_full_result(created_from=day_date, created_until=day_date + relativedelta(days=+1)):
            yield self.get_order_snapshot(row)

    def get_update_snapshots(self, day_date):
        for row in self.get_full_result(last_change=day_date):
            yield self.get_order_snapshot(row)

    def get_result_once(self, **kwargs):
        rsp = self.client.post(
            self.base_url,
            timeout=self.REQUEST_TIMEOUT,
            data=kwargs,
        )
        data = rsp.json()['result']
        return data

    def get_full_result(self, **kwargs):
        rows = self.PAGE_SIZE
        offset = 0
        LOG.debug('Request result at offset %d', offset)
        page_result = self.get_result(rows=rows, offset=offset, **kwargs)
        for row in page_result:
            yield row
        while page_result:
            offset += rows
            LOG.debug('Request result at offset %d', offset)
            page_result = self.get_result(rows=rows, offset=offset, **kwargs)
            for row in page_result:
                yield row

    def get_order_snapshot(self, raw_snapshot):
        order_time = parse_datetime_iso(raw_snapshot['created'])
        partner_status = raw_snapshot['status']

        ttv = raw_snapshot['ttv']
        order_amount = ttv['amount']
        currency = ttv['currency']
        if currency != 'EUR':
            raise RuntimeError('Unexpected currency {}'.format(currency))

        profit_amount = raw_snapshot['euro_fee']

        snapshot = BookingOrderSnapshot()
        snapshot.update_partner_order_id(text_type(raw_snapshot['reservation_id']))
        snapshot.status = self.STATUS_MAPPING[partner_status]
        snapshot.label = raw_snapshot['affiliate_label']
        snapshot.currency_code = OrderCurrencyCode.EUR
        snapshot.order_amount = order_amount
        snapshot.created_at = timestamp(order_time)

        # hotels specific
        snapshot.partner_status = partner_status
        snapshot.check_in = raw_snapshot['checkin']
        snapshot.check_out = raw_snapshot['checkout']
        snapshot.profit_amount = profit_amount
        snapshot.hotel_name = raw_snapshot['hotel_name']
        snapshot.hotel_country = raw_snapshot['hotel_countrycode']
        snapshot.hotel_city = raw_snapshot['destination_name']
        snapshot.partner_hotel_id = text_type(raw_snapshot['hotel_id'])
        return snapshot
