# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date
from dateutil.relativedelta import relativedelta
from six import text_type

from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.lib_datetime import iter_month, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import Hotels101OrderSnapshot, OrderCurrencyCode, OrderStatus


LOG = get_logger(__name__)


class Hotels101Collector(HttpCollector):
    PARTNER_NAME = 'hotels101'
    BASE_URL = 'https://api.101hotels.ru/yandextravelapi/booking/stat'
    REQUEST_TIMEOUT = 60
    STATUS_MAPPING = {
        'progress': OrderStatus.PENDING,
        'cancelled': OrderStatus.REFUNDED,
        'success': OrderStatus.CONFIRMED,
    }

    def __init__(self, options):
        super(Hotels101Collector, self).__init__(options)
        self.auth = (options.username, options.password)
        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()
        self.get_month_orders = with_retries(
            self.get_month_orders_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)
        parser.add_argument('-u', '--username', required=True)
        parser.add_argument('-p', '--password', required=True)
        parser.add_argument('--date-from', default=date(year=2017, month=7, day=1).isoformat())
        parser.add_argument('--date-to', default=(date.today() + relativedelta(years=+2)).isoformat())

    def _get_snapshots(self):
        for month_date in iter_month(self.date_from, self.date_to):
            for snapshot in self.get_month_snapshots(month_date):
                yield snapshot

    def get_month_snapshots(self, month_date):
        LOG.info('Processing {:04}-{:02}'.format(month_date.year, month_date.month))
        for raw_order in self.get_month_orders(month_date):
            yield self.get_order_snapshot(raw_order)

    def get_month_orders_once(self, month_date):
        params = {
            'date_from': month_date,
            'date_to': month_date + relativedelta(months=+1, days=-1)
        }
        rsp = self.request_get(self.base_url, params=params, timeout=self.REQUEST_TIMEOUT, auth=self.auth)
        return rsp.json()

    def get_order_snapshot(self, raw_order):
        if 'commission_amount' not in raw_order and raw_order['status'] != 'cancelled':
            LOG.warn('Order is active but has no "commission_amount" %r', raw_order)
        created_at = parse_datetime_iso(raw_order['booking_date'])
        partner_status = raw_order['status']
        check_in = parse_datetime_iso(raw_order['check_in_date']).date()
        check_out = check_in + relativedelta(days=int(raw_order['nights']))
        order_amount = float(raw_order['booked_amount'])
        profit_amount = raw_order.get('commission_amount', order_amount * 0.09)
        profit_amount = float(profit_amount)
        snapshot = Hotels101OrderSnapshot()
        snapshot.update_partner_order_id(text_type(raw_order['offer_id']))
        snapshot.status = self.STATUS_MAPPING[raw_order['status']]
        snapshot.label = raw_order.get('label')
        snapshot.currency_code = OrderCurrencyCode.RUB
        snapshot.order_amount = order_amount
        snapshot.created_at = timestamp(created_at)

        # hotels specific
        snapshot.partner_status = partner_status
        snapshot.check_in = text_type(check_in)
        snapshot.check_out = text_type(check_out)
        snapshot.profit_amount = profit_amount
        snapshot.hotel_name = raw_order['hotel_name']
        snapshot.hotel_country = raw_order['country_name']
        snapshot.hotel_city = raw_order['region_name']  # TODO: region is not a city

        return snapshot
