# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from six import binary_type
import yt.wrapper as yt

from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.lib_datetime import parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import BookingOrderSnapshot, HotelsCombinedOrderSnapshot, LevelTravelOrderSnapshot
from travel.cpa.lib.order_snapshot import OstrovokOrderSnapshot, OrderCurrencyCode, OrderStatus


LOG = get_logger(__name__)

LABEL_FIELDS = [
    'label_source',
    'label_medium',
    'label_campaign',
    'label_content',
    'label_term',
    'label_uid',
    'label_yuid',
    'label_hotel_id',
    'label_operator_id',
    'label_query',
    'label_req_id',
    'label_price',
    'label_testids',
    'label_cachetimestamp',
    'label_permalink',
    'label_offerid',
    'label_requestid',
    'label_offersource',
    'label_offercampaign',
]


class HotelsHistoryCollector(HttpCollector):
    PARTNER_NAME = 'hotels_history'

    def __init__(self, options):
        super(HotelsHistoryCollector, self).__init__(options)
        self.partner_tables = [
            (BookingOrderSnapshot, '//home/travel/prod/stats/booking'),
            (HotelsCombinedOrderSnapshot, '//home/travel/prod/stats/hotelscombined'),
            (LevelTravelOrderSnapshot, '//home/travel/prod/stats/lt'),
            (OstrovokOrderSnapshot, '//home/travel/prod/stats/ostrovok'),
        ]
        self.labels_table = '//home/travel/prod/stats/labels/labels'
        yt.config['token'] = options.yt_token
        yt.config['proxy']['url'] = options.yt_proxy
        LOG.info('Getting labels map from %s', self.labels_table)
        self.labels_map = self.get_labels_map()
        LOG.info('Got %d labels records', len(self.labels_map))

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

    def get_labels_map(self):
        labels_map = dict()
        for row in yt.read_table(self.labels_table):
            fields = row['Fields']
            tail_size = len(LABEL_FIELDS) - len(fields)
            fields = [f.decode('utf8') for f in fields] + [None] * tail_size
            fields = (None if f == '' else f for f in fields)
            key = tuple(fields)
            labels_map[key] = row['Label'].decode()
        return labels_map

    def _get_snapshots(self):
        for partner_cls, table in self.partner_tables:
            for snapshot in self.get_partner_snapshots(partner_cls, table):
                yield snapshot

    def get_partner_snapshots(self, partner_cls, table):
        LOG.info('Processing {}'.format(table))
        for order in yt.read_table(table, enable_read_parallel=True):
            yield self.get_order_snapshot(order, partner_cls)

    def get_label(self, raw_order):
        all_fields = (raw_order[lf] for lf in LABEL_FIELDS)
        fields = (None if af is None else binary_type(af).decode('utf8') for af in all_fields)
        fields = (None if f == '' else f for f in fields)
        key = tuple(fields)
        label = self.labels_map.get(key)
        if label is None and not raw_order['label_empty']:
            LOG.warn('Label not found for key: %r,', key)
            LOG.warn('for order %r', raw_order)
        return label

    def get_order_snapshot(self, raw_order, partner_cls):
        location = raw_order['location'] or ','
        parts = location.decode('utf8').rsplit(',', 1)
        country, city = parts[:2]

        snapshot = partner_cls()
        snapshot.snapshot_source = 'history'
        snapshot.update_partner_order_id(raw_order['partner_order_id'].decode('utf8'))
        snapshot.status = OrderStatus(raw_order['status'].decode('utf8'))
        snapshot.label = self.get_label(raw_order)
        snapshot.currency_code = OrderCurrencyCode(raw_order['currency_code'].decode('utf8'))
        snapshot.order_amount = raw_order['order_amount']
        snapshot.created_at = timestamp(parse_datetime_iso(raw_order['order_time']))
        snapshot.updated_at = timestamp(parse_datetime_iso(raw_order['update_time']))

        # hotels specific
        snapshot.partner_status = raw_order['partner_status'].decode('utf8') or None
        snapshot.check_in = raw_order['check_in'].decode('utf8')
        snapshot.check_out = raw_order['check_out'].decode('utf8')
        snapshot.profit_amount = raw_order['profit_amount']
        snapshot.hotel_name = raw_order['hotel'].decode('utf8')
        snapshot.hotel_country = country
        snapshot.hotel_city = city

        return snapshot
