# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date

from dateutil.relativedelta import relativedelta
from six import text_type

from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.lib_datetime import parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import HotelsCombinedOrderSnapshot, OrderCurrencyCode, OrderStatus


LOG = get_logger(__name__)


class HotelscombinedCollector(HttpCollector):
    PARTNER_NAME = 'hotelscombined'
    BASE_URL = 'https://api.affiliates.hotelscombined.com/api/transactions/bookings'
    REQUEST_TIMEOUT = 60

    def __init__(self, options):
        super(HotelscombinedCollector, self).__init__(options)
        self.api_key = options.api_key
        self.date_to = date.today()
        # this is max depth we can go to the past and nothing beyond this, only errors
        self.date_from = self.date_to + relativedelta(months=-3, days=+2)
        self.get_raw_snapshots = with_retries(
            self.get_raw_snapshots_once,
            counter=self.metrics,
            key='collector.events.invalid_response',
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)
        parser.add_argument('--api-key', required=True)

    def _get_snapshots(self):
        LOG.info('Getting order snapshots starting from %r', self.date_from)
        for raw_snapshot in self.get_raw_snapshots():
            yield self.get_order_snapshot(raw_snapshot)

    def get_raw_snapshots_once(self):
        params = {
            'apikey': self.api_key,
            'startdate': self.date_from,
            'enddate': self.date_to,
        }
        rsp = self.request_get(self.base_url, params=params, timeout=self.REQUEST_TIMEOUT)
        return rsp.json()

    def get_order_snapshot(self, raw_snapshot):
        if 'commission' not in raw_snapshot:
            LOG.warn('No "commission" in raw snapshot: %r', raw_snapshot)

        LOG.info(raw_snapshot)
        created_at = parse_datetime_iso(raw_snapshot['BookingDate'])
        check_in = parse_datetime_iso(raw_snapshot['BookingCheckinDate']).date()
        check_out = parse_datetime_iso(raw_snapshot['BookingCheckoutDate']).date()

        snapshot = HotelsCombinedOrderSnapshot()
        snapshot.update_partner_order_id(text_type(raw_snapshot['BookingID']))
        snapshot.status = OrderStatus.CONFIRMED
        snapshot.label = raw_snapshot['Label']
        snapshot.currency_code = OrderCurrencyCode.USD
        snapshot.order_amount = float(raw_snapshot['Revenue'])
        snapshot.created_at = timestamp(created_at)

        # hotels specific
        snapshot.partner_status = None
        snapshot.check_in = text_type(check_in)
        snapshot.check_out = text_type(check_out)
        commission = raw_snapshot.get('Commission')
        if commission is None:
            commission = snapshot.order_amount * 0.75
        snapshot.profit_amount = float(commission)
        snapshot.hotel_name = raw_snapshot['HotelName']
        snapshot.hotel_country = raw_snapshot['HotelCountry']
        snapshot.hotel_city = raw_snapshot['HotelCity']
        snapshot.partner_hotel_id = text_type(raw_snapshot['HotelID'])

        if raw_snapshot['HotelName'] is None:
            LOG.warn('Field "HotelName" not assigned for %s', snapshot.partner_order_id)

        return snapshot
