# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date
from dateutil.relativedelta import relativedelta
from six import iteritems, text_type
import csv
import io

from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.lib_datetime import iter_month, parse_datetime_iso
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import LevelTravelOrderSnapshot, OrderCurrencyCode, OrderStatus


LOG = get_logger(__name__)


class UnicodeDictReader(csv.DictReader, object):
    def __init__(self, f, encoding, **kwargs):
        super(UnicodeDictReader, self).__init__(f, **kwargs)
        self.encoding = encoding

    @staticmethod
    def decode_value(item, encoding):
        if isinstance(item, unicode):
            return item
        if isinstance(item, list):
            return [text_type(i, encoding) for i in item]
        return text_type(item, encoding)

    def next(self):
        row = super(UnicodeDictReader, self).next()
        encoding = self.encoding
        return {self.decode_value(key, encoding): self.decode_value(value, encoding) for key, value in iteritems(row)}


class LevelTravelCollector(HttpCollector):
    PARTNER_NAME = 'leveltravel'
    BASE_URL = 'http://api.level.travel/reports/orders.tsv'
    REQUEST_TIMEOUT = 60
    STATUS_MAPPING = {
        'created': OrderStatus.PENDING,
        'booked': OrderStatus.PENDING,
        'approved': OrderStatus.PENDING,
        'canceled': OrderStatus.CANCELLED,
        'cancelled': OrderStatus.CANCELLED,
        'not_confirmed': OrderStatus.CANCELLED,
    }

    def __init__(self, options):
        super(LevelTravelCollector, self).__init__(options)
        self.api_key = options.api_key
        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()
        self.get_month_orders = with_retries(
            self.get_month_orders_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)
        parser.add_argument('--api-key', required=True)
        parser.add_argument('--date-from', default=date(year=2015, month=1, day=1).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

    def _get_snapshots(self):
        for month_date in iter_month(self.date_from, self.date_to):
            for snapshot in self.get_month_snapshots(month_date):
                yield snapshot

    def get_month_snapshots(self, month_date):
        LOG.info('Processing {:04}-{:02}'.format(month_date.year, month_date.month))
        month_orders, encoding = self.get_month_orders(month_date)
        f = io.BytesIO(month_orders)
        reader = UnicodeDictReader(f, encoding=encoding, restkey='others', dialect='excel-tab')
        for row in reader:
            yield self.get_order_snapshot(row)

    def get_month_orders_once(self, month_date):
        params = {
            'key': self.api_key,
            'start_date': month_date,
            'end_date': month_date + relativedelta(months=+1, days=-1),
        }
        rsp = self.request_get(self.base_url, params=params, timeout=self.REQUEST_TIMEOUT)
        return rsp.content, rsp.encoding

    def get_order_snapshot(self, raw_order):
        partner_status = raw_order['state']
        check_in = parse_datetime_iso(raw_order['start_date']).date()
        check_out = check_in + relativedelta(days=int(raw_order['nights']))

        snapshot = LevelTravelOrderSnapshot()
        snapshot.update_partner_order_id(raw_order['id'])
        snapshot.status = self.STATUS_MAPPING.get(partner_status, OrderStatus.CONFIRMED)
        snapshot.label = raw_order.get('others', [None])[0]
        snapshot.currency_code = OrderCurrencyCode.RUB
        snapshot.order_amount = float(raw_order['price'])
        snapshot.created_at = int(raw_order['created_at'])

        # hotels specific
        snapshot.partner_status = partner_status
        snapshot.check_in = text_type(check_in)
        snapshot.check_out = text_type(check_out)
        snapshot.profit_amount = float(raw_order['comission'])
        snapshot.hotel_name = raw_order['hotel']
        snapshot.hotel_country = raw_order['country']
        snapshot.hotel_city = raw_order['hotel_city']

        return snapshot
