# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date, timedelta
import calendar

from dateutil.relativedelta import relativedelta
from six import text_type
from six.moves.urllib.parse import urlparse, parse_qs

from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.lib_datetime import parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderStatus, OrderCurrencyCode, OstrovokOrderSnapshot


LOG = get_logger(__name__)


class OstrovokCollector(HttpCollector):
    PARTNER_NAME = 'ostrovok'
    BASE_URL = 'https://dapi.worldota.net/api/partner/orders'
    REQUEST_TIMEOUT = 90
    PAGE_SIZE = 100
    STATUS_MAPPING = {
        'noshow': OrderStatus.REFUNDED,
        'cancelled': OrderStatus.REFUNDED,
        'rejected': OrderStatus.REFUNDED,
        'failed': OrderStatus.REFUNDED,
        'completed': OrderStatus.PENDING,
    }

    def __init__(self, options):
        super(OstrovokCollector, self).__init__(options)
        self.auth = (options.username, options.password)
        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = (parse_datetime_iso(options.date_to).date()) + relativedelta(days=+1)
        if options.days_before is not None:
            self.date_from = self.date_to - relativedelta(days=options.days_before)
        self.get_page = with_retries(
            self.get_page_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)
        parser.add_argument('-u', '--username', required=True)
        parser.add_argument('-p', '--password', required=True)
        parser.add_argument('--date-from', default=(date.today() - relativedelta(days=7)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())
        parser.add_argument('--days-before', type=int)

    @staticmethod
    def float_or_zero(s):
        return float(s) if s else 0.0

    @staticmethod
    def msk_to_utc(dt):
        return dt - timedelta(hours=3)

    def get_status(self, partner_status, checkout_date):
        status = self.STATUS_MAPPING[partner_status]
        if status == OrderStatus.REFUNDED:
            return status
        checkout_date = parse_datetime_iso(checkout_date).date()
        with_last_day = lambda y, m: date(year=y, month=m, day=calendar.monthrange(y, m)[1])
        if checkout_date.month == 12:
            payment_due_date = with_last_day(checkout_date.year + 1, 1)
        else:
            payment_due_date = with_last_day(checkout_date.year, checkout_date.month + 1)
        return OrderStatus.CONFIRMED if payment_due_date <= self.date_to else OrderStatus.PENDING

    def get_page_once(self, page_num):
        params = {
            'created_from': self.date_from,
            'created_to': self.date_to + relativedelta(days=+1),
            'limit': self.PAGE_SIZE,
            'page': page_num,
            'lang': 'ru',
        }
        rsp = self.request_get(self.base_url, timeout=self.REQUEST_TIMEOUT, auth=self.auth, params=params)
        return rsp.json()

    def get_raw_orders(self):
        page_num = 1
        while True:
            LOG.info('Processing page %d', page_num)
            page = self.get_page(page_num)
            raw_orders = page.get('orders', [])
            if not raw_orders:
                break
            for raw_order in raw_orders:
                yield raw_order
            page_num += 1

    def _get_snapshots(self):
        for order in self.get_raw_orders():
            utm_term = None
            if order['partner_original_url']:
                query = urlparse(order['partner_original_url']).query
                utm_term = parse_qs(query).get('partner_extra_cookie', [None])[0]
            partner_status = order['status']

            snapshot = OstrovokOrderSnapshot()
            snapshot.update_partner_order_id(text_type(order['order_id']))
            snapshot.status = self.get_status(partner_status, order['checkout_date'])
            snapshot.label = utm_term
            snapshot.currency_code = OrderCurrencyCode.RUB
            snapshot.order_amount = float(order['amount'])
            snapshot.created_at = timestamp(self.msk_to_utc(parse_datetime_iso(order['created_at'])))

            # hotels specific
            snapshot.partner_status = partner_status
            snapshot.check_in = order['checkin_date']
            snapshot.check_out = order['checkout_date']
            snapshot.profit_amount = self.float_or_zero(order['partner_commission'])
            snapshot.hotel_name = order.get('hotel_name') or ''
            snapshot.hotel_country = order.get('country')
            snapshot.hotel_city = order.get('city')
            snapshot.partner_hotel_id = order.get('hotel_master_id')

            if snapshot.hotel_name == '':
                LOG.warn('Field "hotel_name" not assigned for %s', snapshot.partner_order_id)

            yield snapshot
