# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json

from datetime import date

from dateutil.relativedelta import relativedelta
from lxml import etree
from six import text_type

from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import iter_periods, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, TvilOrderSnapshot


LOG = get_logger(__name__)


class TvilCollector(HttpCollector):
    PARTNER_NAME = 'tvil'
    BASE_URL = 'https://chm.tvil.ru/yandexTravel'
    BASE_URL_TESTING = 'https://chm-kakugodno.tvil.ru/yandexTravel'
    REQUEST_TIMEOUT = 90
    STATUS_MAPPING = {
        'UNPAID': OrderStatus.UNPAID,
        'CONFIRMED': OrderStatus.PENDING,
        'FINISHED': OrderStatus.CONFIRMED,
        'CANCELLED': OrderStatus.CANCELLED,
        'REFUNDED': OrderStatus.REFUNDED,
    }

    def __init__(self, options):
        super(TvilCollector, self).__init__(options)
        self.auth = options.password
        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = (parse_datetime_iso(options.date_to).date()) + relativedelta(days=+1)
        self.single_report_days = int(options.single_report_days)
        self.get_report = with_retries(
            self.get_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)
        parser.add_argument('-p', '--password', required=True)
        parser.add_argument('--date-from', default=date(year=2020, month=5, day=4).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())
        parser.add_argument('--single-report-days', default='14')

    def get_report_once(self, period_from, period_to):
        LOG.info('Requesting data for period [{}, {})'.format(period_from.isoformat(), period_to.isoformat()))
        params = {
            'action': 'get-bookings',
            'partner': 'yatravel',
            'password': self.auth,
            'data': {
                'date1': period_from.isoformat() + ' 00:00:00',
                'date2': period_to.isoformat() + ' 00:00:00',
            },
        }
        rsp = self.request_post(self.base_url, timeout=self.REQUEST_TIMEOUT, data=json.dumps(params))
        return rsp.text

    def get_single_report_raw_orders(self, period_from, period_to):
        report_xml_text = self.get_report(period_from, period_to)
        try:
            report_xml_tree = etree.fromstring(report_xml_text, parser=etree.XMLParser(encoding='utf-8'))
        except etree.XMLSyntaxError:
            LOG.exception('Bad Tvil XML')
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        return report_xml_tree.xpath('/bookings/booking')

    def get_all_raw_orders(self):
        for period_from, period_to in iter_periods(self.date_from, self.date_to, self.single_report_days):
            for order_xml_tree in self.get_single_report_raw_orders(period_from, period_to):
                yield order_xml_tree

    def _get_snapshots(self):
        for order_xml_tree in self.get_all_raw_orders():
            order_dict = {text_type(e.tag): text_type(e.text) for e in order_xml_tree}
            partner_status = order_dict['state']

            snapshot = TvilOrderSnapshot()
            snapshot.update_partner_order_id(order_dict['id'])
            snapshot.status = self.STATUS_MAPPING[partner_status]
            snapshot.label = order_dict['marker']
            snapshot.currency_code = OrderCurrencyCode(order_dict['currency'])
            snapshot.order_amount = float(order_dict['price'])
            snapshot.created_at = timestamp(parse_datetime_iso(order_dict['createdAt']))

            # hotels specific
            snapshot.partner_status = partner_status
            snapshot.check_in = order_dict['checkInDate']
            snapshot.check_out = order_dict['checkOutDate']
            snapshot.hotel_name = ''
            snapshot.partner_hotel_id = order_dict['hotelId']

            yield snapshot
