# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json
from datetime import date

import six
from yt.wrapper import YtClient, format

from travel.cpa.collectors.lib.collector import Collector
from travel.cpa.lib.lib_datetime import parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus

LOG = get_logger(__name__)


class AviaPartnerBookingLogCollector(Collector):
    STATUS_MAPPING = {
        'booking': OrderStatus.PENDING,
        'paid': OrderStatus.CONFIRMED,
        'cancel': OrderStatus.CANCELLED,
    }

    SNAPSHOT_CLS = None
    CURRENCY_INSTEAD_OF_NONE = OrderCurrencyCode.RUB.value

    def __init__(self, options, partner_name):
        super(AviaPartnerBookingLogCollector, self).__init__()

        self.booking_dir = options.booking_dir
        self.snapshot_cls = self.SNAPSHOT_CLS
        self.partner_name = partner_name
        yt_client = YtClient(options.yt_proxy, options.yt_token)
        self.yt_client = yt_client

    def map(self, row):
        if row['partner'] == self.partner_name:
            yield row

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--booking-dir', help='Full yt path to avia_partner_booking_log/ directory', required=True)

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

    def _get_snapshots(self):
        snapshots_by_order_id = {}
        for row in self._get_rows():
            snapshot = self.row_to_snapshot(row)
            if snapshot is not None:
                self._put_or_replace_with_most_recent(snapshots_by_order_id, snapshot)

        for snapshot in snapshots_by_order_id.values():
            yield snapshot

    @staticmethod
    def _put_or_replace_with_most_recent(snapshots, snapshot):
        if snapshot.travel_order_id not in snapshots:
            snapshots[snapshot.travel_order_id] = snapshot
            return

        current_snapshot = snapshots[snapshot.travel_order_id]
        if current_snapshot.source_updated_at < snapshot.source_updated_at:
            snapshots[snapshot.travel_order_id] = snapshot
            return

    def _get_rows(self):
        with self.yt_client.TempTable() as filtered_by_partners:
            input_tables = list(self.yt_client.search(self.booking_dir, node_type='table',
                                                      path_filter=self._filter_yt_tables))
            self.yt_client.run_map(self.map, input_tables, filtered_by_partners)

            rows = self.yt_client.read_table(filtered_by_partners, format=format.JsonFormat())
            for row in rows:
                yield row

    def _filter_yt_tables(self, path):
        return True

    def row_to_snapshot(self, row):
        required_columns = {'order_id', 'status', 'created_at', 'partner_id', 'billing_order_id'}
        if any([column not in row for column in required_columns]):
            LOG.error(
                'Row doesnt contain required columns (%s): %s',
                ','.join([column for column in required_columns if column not in row]),
                row,
            )
            return None

        if not all(row[column] for column in required_columns):
            LOG.error(
                'Some required columns are empty (%s): %s',
                ','.join([column for column in required_columns if not row[column]]),
                row,
            )
            return None

        order = {
            'status': self.STATUS_MAPPING[row['status']],
            'label': six.text_type(row.get('marker')),
            'created_at': timestamp(parse_datetime_iso(row['created_at'])),
            'order_amount': float(row.get('price') or 0.0),
            'currency_code': OrderCurrencyCode(row.get('currency') or self.CURRENCY_INSTEAD_OF_NONE),
            'partner_id': row['partner_id'],
            'billing_order_id': row['billing_order_id'],
            'origin': six.text_type(row.get('from')) if row.get('from') else None,
            'destination': six.text_type(row.get('to')) if row.get('to') else None,
            'trip_type': six.text_type(row.get('trip_type')) if row.get('trip_type') else None,
            'source_updated_at': row['unixtime'],  # время последнего обновления записи в источнике
        }
        snapshot = self.snapshot_cls.from_dict(
            d=order,
            convert_type=False,
            ignore_unknown=True,
        )
        snapshot.update_partner_order_id(six.text_type(row['order_id']))
        return snapshot


class AviaPartnerBookingLogHistoryCollector(AviaPartnerBookingLogCollector):
    """
    Базовый класс для переноса истории из avia-partner-booking-log

    Первым этапом можно в наследнике не переопределять логику и собрать все ошибки данных:
    ./collectors_exec --partner-name ozon_history --lb-skip-send
            --booking-dir //home/avia/logs/avia-partner-booking-log --yt-token your-yt-token
    Вторым этапом, при необходимости, можно переопределить метод row_to_snapshot
    """
    SOURCE = 'history'
    CURRENCY_INSTEAD_OF_NONE = None

    def __init__(self, options, partner_name, partner_name_in_log=None):
        super(AviaPartnerBookingLogHistoryCollector, self).__init__(options, partner_name)

        self.partner_name_in_log = partner_name_in_log or partner_name

        self.date_from = options.date_from
        self.date_to = options.date_to

    @classmethod
    def configure(cls, parser):
        """
        Дата старта (--date-from) 2018-08-31 https://st.yandex-team.ru/RASPTICKETS-16502#5e7240cf66454959c9421686
        """
        super(AviaPartnerBookingLogHistoryCollector, cls).configure(parser)

        parser.add_argument('--date-from', default=date(2018, 8, 31).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

    def map(self, row):
        if row['partner'] == self.partner_name_in_log:
            yield row

    def _filter_yt_tables(self, path):
        return self.date_from <= path.rsplit('/', 1)[-1] <= self.date_to

    def _get_snapshots(self):
        for row in self._get_rows():
            try:
                yield self.row_to_snapshot(row)
            except Exception as e:
                for k in row:
                    if isinstance(row[k], six.string_types):
                        row[k] = six.ensure_text(row[k])
                LOG.exception('%s %s', e, json.dumps(row, sort_keys=True, indent=2, ensure_ascii=False))

    def row_to_snapshot(self, row):
        raw_currency_code = row['currency']
        if raw_currency_code == 'RUR':
            raw_currency_code = OrderCurrencyCode.RUB.value
        if not raw_currency_code and self.CURRENCY_INSTEAD_OF_NONE:
            raw_currency_code = self.CURRENCY_INSTEAD_OF_NONE
        order = {
            'status': self.STATUS_MAPPING[row['status']],
            'label': self.to_unicode_or_none(row['marker']),
            'created_at': timestamp(parse_datetime_iso(row['created_at'])),
            'updated_at': timestamp(self.now),
            'order_amount': float(row['price']) if row['price'] is not None else None,
            'currency_code': OrderCurrencyCode(raw_currency_code),
            'partner_id': row['partner_id'],
            'billing_order_id': row['billing_order_id'],
            'origin': self.to_unicode_or_none(row['from']),
            'destination': self.to_unicode_or_none(row['to']),
            'trip_type': self.to_unicode_or_none(row.get('trip_type')),
            'source': self.SOURCE,
        }
        snapshot = self.snapshot_cls.from_dict(
            d=order,
            convert_type=False,
            ignore_unknown=True,
        )
        snapshot.update_partner_order_id(self.to_unicode_or_none(row['order_id']))
        snapshot.validate()  # При ошибке падаем здесь, чтобы обработать другие строчки лога
        return snapshot

    @staticmethod
    def to_unicode_or_none(value):
        if value is None:
            return value
        return six.ensure_text(value)
