# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from copy import copy
from datetime import date, datetime, timedelta

from yt.wrapper import YtClient
import ujson as json

from dateutil.relativedelta import relativedelta
from travel.avia.library.python.marker_helpers import AviaFlightRouteHelper
from travel.avia.library.python.references.partner import PartnerCache
from travel.avia.library.python.references.station import create_station_cache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries, abstract_attribute
from travel.cpa.lib.errors import ProcessError, ErrorType
from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus

LOG = get_logger(__name__)


class AviaJsonCollector(HttpCollector):
    REQUEST_TIMEOUT = 60

    PARTNER_NAME = abstract_attribute
    PARTNER_CODE = abstract_attribute

    SNAPSHOT_CLS = abstract_attribute

    STATUS_MAPPING = {
        'W': OrderStatus.PENDING,
        'NP': OrderStatus.PENDING,
        'NC': OrderStatus.PENDING,
        'CM': OrderStatus.PENDING,
        'WA': OrderStatus.PENDING,
        'P': OrderStatus.CONFIRMED,
        'PN': OrderStatus.CONFIRMED,
        'CR': OrderStatus.CONFIRMED,
        'R': OrderStatus.CONFIRMED,
        'OR': OrderStatus.CONFIRMED,
        'LCP': OrderStatus.CONFIRMED,
        'PC': OrderStatus.CANCELLED,
        'C': OrderStatus.CANCELLED,
        'CO': OrderStatus.CANCELLED,
        'CU': OrderStatus.CANCELLED,
        'CL': OrderStatus.CANCELLED,
        'WD': OrderStatus.CANCELLED,
        'Cancelled': OrderStatus.CANCELLED,
    }

    def __init__(self, options):
        super(AviaJsonCollector, self).__init__(options)

        self.auth = None
        self.params = None

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(self.PARTNER_CODE)

        self.station = create_station_cache(yt_client)
        self.route_helper = AviaFlightRouteHelper(self.station)

        self.get_day_report = with_retries(
            self.get_day_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        self.auth = (options.username, options.password)
        self.params = {
            'key': options.userkey,
            'user_token': options.usertoken,
            'gmt': '0',
        }

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-4)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('--username', default=None)
        parser.add_argument('--password', default=None)
        parser.add_argument('--userkey', default=None)
        parser.add_argument('--usertoken', default=None)

    def _get_snapshots(self):
        for day_date in iter_day(self.date_from, self.date_to):
            LOG.info('Getting snapshots for %r', day_date)
            for snapshot in self.get_day_snapshots(day_date):
                if snapshot is None:
                    continue
                yield snapshot

    def get_day_snapshots(self, day_date):
        report = self.get_day_report(day_date)
        try:
            raw_data = self.ensure_correct_jsonrpc(report)
            tree = json.loads(raw_data)
            for raw_snapshot in tree['items']:
                yield self.get_order_snapshot(raw_snapshot)
        except ValueError:
            LOG.exception('Bad %s JSON %s', self.PARTNER_CODE.upper(), report[:1000])
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

    def get_day_report_once(self, day_date):
        params = copy(self.params)
        params.update({
            'date_from': (day_date - timedelta(days=1)).strftime('%d.%m.%Y'),
            'date_to': (day_date + timedelta(days=1)).strftime('%d.%m.%Y'),
        })
        rsp = self.request_post(self.base_url, params=params, auth=self.auth, timeout=self.REQUEST_TIMEOUT)
        return rsp.content

    def ensure_correct_jsonrpc(self, raw_response):
        # type: (str) -> str
        json_rpc_data = json.loads(raw_response)
        if 'json' not in json_rpc_data:
            raise ValueError('Incorrect jsonrpc data %s' % json_rpc_data)
        if not '200' == json_rpc_data['json'].get('status'):
            raise ValueError('Bad jsonrpc status: %s' % json_rpc_data['json'].get('status'))
        return json_rpc_data['json'].get('result')

    def get_order_snapshot(self, order_dict):
        try:
            flights = map(self.get_mapped_flight, order_dict['locations'])
            flights = sorted(flights, key=lambda flight: flight['departure_dt'])
            route_info = self.route_helper.fillin_trip_info({'flights': flights})
            snapshot = self.SNAPSHOT_CLS.from_dict(route_info, ignore_unknown=True)
        except:
            LOG.exception('Could not get route information')
            snapshot = self.SNAPSHOT_CLS()

        snapshot.update_partner_order_id(order_dict.get('locator'))
        snapshot.status = self.STATUS_MAPPING[order_dict.get('status')]
        snapshot.label = order_dict.get('marker')
        snapshot.created_at = order_dict.get('date')
        snapshot.updated_at = timestamp(self.now)
        snapshot.order_amount = float(order_dict['final_all_price']['RUR'])
        snapshot.currency_code = OrderCurrencyCode('RUB')
        snapshot.adults = order_dict.get('adults')
        snapshot.children = order_dict.get('children')
        snapshot.infants = order_dict.get('infants')
        snapshot.partner_id = self.partner_id
        snapshot.billing_order_id = self.billing_order_id

        LOG.debug(snapshot.as_dict())
        return snapshot

    @staticmethod
    def get_mapped_flight(flight):
        return {
            'from': flight.get('departure_location'),
            'to': flight.get('arrival_location'),
            'departure_dt': datetime.fromtimestamp(flight.get('departure_time')),
            'arrival_dt': datetime.fromtimestamp(flight.get('arrival_time')),
        }
