# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from dateutil.relativedelta import relativedelta
from datetime import date, datetime
from six import text_type
from abc import abstractmethod

from ticket_parser2.api.v1 import TvmClient, TvmClientStatus, TvmApiClientSettings
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, DeferredPaymentEligibility

LOG = get_logger(__name__)


class BoYHotelCollector(HttpCollector):
    ORDER_TYPE = None
    BASE_URL = 'http://api-prod.travel-hotels.yandex.net/api/cpa_export/v2/get_cpa_order_snapshots'
    EXTENDED_STATUS_MAPPING = {
        'UNPAID': OrderStatus.UNPAID,
        'CONFIRMED': OrderStatus.CONFIRMED,
        'CANCELLED': OrderStatus.CANCELLED,
        'REFUNDED': OrderStatus.REFUNDED,
        'PAID': OrderStatus.PAID
    }
    DEFERRED_ELIGIBILITY_MAPPING = {
        'UNKNOWN': DeferredPaymentEligibility.UNKNOWN,
        'ELIGIBLE': DeferredPaymentEligibility.ELIGIBLE,
        'NON_ELIGIBLE': DeferredPaymentEligibility.NON_ELIGIBLE,
    }

    def __init__(self, options):
        super(BoYHotelCollector, self).__init__(options)
        use_tvm = options.use_tvm

        tvm_ticket = None
        if use_tvm:
            tvm_secret = options.tvm_secret
            tvm_client_id = options.tvm_client_id
            tvm_service_id = options.tvm_service_id
            tvm_client = self.create_tvm_client(tvm_service_id, tvm_client_id, tvm_secret)
            tvm_ticket = tvm_client.get_service_ticket_for('cpa_export_api')
        LOG.debug('Got tvm ticket: {}'.format(tvm_ticket))
        self.tvm_ticket = tvm_ticket

        self.updated_at_from = datetime.combine(parse_datetime_iso(options.date_from).date(), datetime.min.time())
        self.updated_at_to = datetime.combine(parse_datetime_iso(options.date_to).date(), datetime.max.time())
        self.limit = options.limit

        self.get_raw_snapshots = with_retries(
            self.get_raw_snapshots_once,
            counter=self.metrics,
            key='collector.events.invalid_response',
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('--use-tvm', action='store_true')
        parser.add_argument('--tvm-secret')
        parser.add_argument('--tvm-client-id', type=int)
        parser.add_argument('--tvm-service-id', type=int)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(days=-7)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())
        parser.add_argument('--limit', default=300)

    @staticmethod
    def create_tvm_client(tvm_service_id, tvm_client_id, tvm_secret):
        destinations = {'cpa_export_api': tvm_service_id}
        settings = TvmApiClientSettings(
            self_client_id=tvm_client_id,
            enable_service_ticket_checking=True,
            enable_user_ticket_checking=False,
            self_secret=tvm_secret,
            dsts=destinations,
        )
        client = TvmClient(settings)
        if client.status != TvmClientStatus.Ok:
            LOG.exception('Bad tvm client status: %s', client.status)
            raise ProcessError(ErrorType.ET_BAD_TVM_CLIENT_STATUS)
        return client

    def _get_snapshots(self):
        LOG.info('Getting order snapshots starting from %r', self.updated_at_from)
        for raw_snapshot_batch in self.get_raw_snapshots_batch():
            for raw_snapshot in raw_snapshot_batch:
                yield self.get_order_snapshot(raw_snapshot)

    def get_raw_snapshots_batch(self):
        has_more = True
        while has_more:
            rsp = self.get_raw_snapshots()
            response = rsp.json()['order_snapshots']
            has_more = rsp.json()['has_more']
            if has_more:
                LOG.info('Will get one more page')
                self.updated_at_from = parse_datetime_iso(response[-1]['updated_at'])
            yield response

    def get_raw_snapshots_once(self):
        params = {
            'order_type': self.ORDER_TYPE,
            'updated_at_from_utc': self.updated_at_from.isoformat(),
            'updated_at_to_utc': self.updated_at_to.isoformat(),
            'limit': self.limit,
        }
        headers = {}
        if self.tvm_ticket is not None:
            headers['X-Ya-Service-Ticket'] = self.tvm_ticket

        rsp = self.request_get(self.base_url, params=params, headers=headers)
        return rsp

    @abstractmethod
    def init_hotel_snapshot(self, raw_snapshot):
        pass

    def get_order_snapshot(self, raw_snapshot):
        currency = raw_snapshot['amount']['currency']

        created_at = parse_datetime_iso(raw_snapshot['created_at'])
        check_in = parse_datetime_iso(raw_snapshot['check_in_date']).date()
        check_out = parse_datetime_iso(raw_snapshot['check_out_date']).date()

        snapshot = self.init_hotel_snapshot(raw_snapshot)

        LOG.info('partner_order_id = "%s"', raw_snapshot['partner_order_id'])
        snapshot.update_partner_order_id(raw_snapshot['partner_order_id'])
        # snapshot.travel_order_id = raw_snapshot['travel_order_id']
        snapshot.status = self.EXTENDED_STATUS_MAPPING[raw_snapshot['order_status']]
        snapshot.label = raw_snapshot['label']

        currency_code = OrderCurrencyCode(currency)
        if currency_code != OrderCurrencyCode.RUB:
            raise ProcessError(ErrorType.ET_PARTNER_DATA)
        snapshot.currency_code = currency_code

        snapshot.order_amount = float(raw_snapshot['amount']['value'])
        profit = raw_snapshot.get('profit')
        if profit is not None:
            snapshot.profit_amount = float(profit['value'])
        snapshot.created_at = timestamp(created_at)

        # hotels specific
        # TODO (mbobrov): think of exporting raw order status from order service
        # snapshot.partner_status = text_type(partner_status)
        snapshot.check_in = text_type(check_in)
        snapshot.check_out = text_type(check_out)
        snapshot.hotel_name = raw_snapshot['hotel_name']
        if 'hotel_country_name' in raw_snapshot:
            snapshot.hotel_country = raw_snapshot['hotel_country_name']
        if 'hotel_city_name' in raw_snapshot:
            snapshot.hotel_city = raw_snapshot['hotel_city_name']
        snapshot.permalink = raw_snapshot['permalink']

        if len(raw_snapshot.get('promo_codes', [])) > 0:
            snapshot.promo_codes = ','.join(raw_snapshot['promo_codes'])
        if len(raw_snapshot.get('promo_actions', [])) > 0:
            snapshot.promo_actions = ','.join(raw_snapshot['promo_actions'])
        if raw_snapshot.get('discount_amount'):
            # ensure that discount and amount have the same currency
            if currency != raw_snapshot['discount_amount']['currency']:
                raise ProcessError(ErrorType.ET_PARTNER_DATA)
            snapshot.discount_amount = float(raw_snapshot['discount_amount']['value'])
        if raw_snapshot.get('promocode_discount_amount'):
            # ensure that discount and amount have the same currency
            if currency != raw_snapshot['promocode_discount_amount']['currency']:
                raise ProcessError(ErrorType.ET_PARTNER_DATA)
            snapshot.promocode_discount_amount = float(raw_snapshot['promocode_discount_amount']['value'])
        if raw_snapshot.get('strike_through_discount_amount'):
            # ensure that discount and amount have the same currency
            if currency != raw_snapshot['strike_through_discount_amount']['currency']:
                raise ProcessError(ErrorType.ET_PARTNER_DATA)
            snapshot.strike_through_discount_amount = float(raw_snapshot['strike_through_discount_amount']['value'])
        snapshot.privetmir_oct2020_eligible = raw_snapshot.get('privetmir_oct2020_eligible', False)
        snapshot.use_deferred_payment = raw_snapshot.get('uses_deferred_payment', False)
        snapshot.refund_reason = raw_snapshot.get('refund_reason')
        snapshot.post_pay_eligible = raw_snapshot.get('post_pay_eligible', False)
        snapshot.post_pay_used = raw_snapshot.get('post_pay_used', False)

        order_amount_payable = raw_snapshot.get('amount_payable')
        if order_amount_payable is not None:
            if currency != order_amount_payable['currency']:
                raise ProcessError(ErrorType.ET_PARTNER_DATA)
            snapshot.order_amount_payable = float(order_amount_payable['value'])

        amount_received_from_user = raw_snapshot.get('amount_received_from_user')
        if amount_received_from_user is not None:
            if currency != amount_received_from_user['currency']:
                raise ProcessError(ErrorType.ET_PARTNER_DATA)
            snapshot.amount_received_from_user = float(amount_received_from_user["value"])
        snapshot.deferred_payment_eligibility = self.DEFERRED_ELIGIBILITY_MAPPING.get(raw_snapshot.get('deferred_payment_eligibility', 'UNKNOWN'), DeferredPaymentEligibility.UNKNOWN)

        initial_payment_amount = raw_snapshot.get('initial_payment_amount')
        if initial_payment_amount is not None:
            if currency != initial_payment_amount['currency']:
                raise ProcessError(ErrorType.ET_PARTNER_DATA)
            snapshot.initial_payment_amount = float(initial_payment_amount['value'])
        penalty_amount_if_unpaid = raw_snapshot.get('penalty_amount_if_unpaid')
        if penalty_amount_if_unpaid is not None:
            if currency != penalty_amount_if_unpaid['currency']:
                raise ProcessError(ErrorType.ET_PARTNER_DATA)
            snapshot.penalty_amount_if_unpaid = float(penalty_amount_if_unpaid['value'])
        last_payment_scheduled_at = raw_snapshot.get('last_payment_scheduled_at')
        if last_payment_scheduled_at is not None:
            snapshot.last_payment_scheduled_at = timestamp(parse_datetime_iso(last_payment_scheduled_at))
        fully_paid_at = raw_snapshot.get('fully_paid_at')
        if fully_paid_at is not None:
            snapshot.fully_paid_at = timestamp(parse_datetime_iso(fully_paid_at))

        yandex_plus = raw_snapshot.get('yandex_plus_cpa_info')
        if yandex_plus is not None:
            snapshot.yandex_plus_mode = yandex_plus.get('mode')
            snapshot.yandex_plus_withdraw_points = yandex_plus.get('withdraw_points')
            snapshot.yandex_plus_topup_points = yandex_plus.get('topup_points')
            topup_date = yandex_plus.get('topup_date')
            if topup_date is not None:
                snapshot.yandex_plus_topup_date = timestamp(parse_datetime_iso(topup_date))
            snapshot.yandex_plus_user_balance = yandex_plus.get('user_balance')

        white_label = raw_snapshot.get('white_label_cpa_info')
        if white_label is not None:
            snapshot.white_label_points_type = white_label.get('points_type')
            snapshot.white_label_points_amount = white_label.get('points_amount')
            snapshot.white_label_customer_number = white_label.get('customer_number')

        return snapshot
