# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from abc import ABCMeta, abstractmethod

import requests
from six.moves.urllib.parse import urlparse, urlunparse

from travel.cpa.collectors.lib.collector import Collector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.lib_logging import get_logger
from travel.library.python.tools import get_size_in_units

LOG = get_logger(__name__)


class HttpCollector(Collector):
    __metaclass__ = ABCMeta
    ALLOW_PROXY = True
    BASE_URL = None
    BASE_URL_TESTING = None
    SET_HOST_HEADER = False

    def __init__(self, options):
        super(HttpCollector, self).__init__()

        self.request_get = self.metered_get_with_retries(
            requests.get, counter=self.metrics, key='collector.events.http_retry'
        )
        self.request_post = self.metered_get_with_retries(
            requests.post, counter=self.metrics, key='collector.events.http_retry'
        )
        self.base_url = options.base_url

        if options.no_proxy:
            return
        if not self.ALLOW_PROXY:
            raise Exception("Proxy is not allowed for this collector. You must specify --no-proxy to run collector.")
        proxy_parts = urlparse(options.proxy_url)
        url_parts = urlparse(options.base_url)
        url_parts = url_parts._replace(scheme=proxy_parts.scheme)
        url_parts = url_parts._replace(netloc=proxy_parts.netloc)
        url_parts = url_parts._replace(
            path='/{}{}{}'.format(self.PARTNER_NAME, options.proxy_path_suffix, url_parts.path)
        )
        LOG.debug('url_parts = %r', url_parts)
        self.base_url = urlunparse(url_parts)
        LOG.debug('base_url = %s', self.base_url)

    @classmethod
    def configure_all(cls, parser):
        parser.add_argument('--proxy-url', default='http://partners-proxy.production.avia.yandex.net')
        parser.add_argument('--proxy-path-suffix', default='')
        parser.add_argument('--no-proxy', action='store_true')

    @classmethod
    @abstractmethod
    def configure(cls, parser):
        """наследники HttpCollector обязаны объявить здесь --base-url"""
        pass

    @abstractmethod
    def _get_snapshots(self):
        pass

    @staticmethod
    def metered_get_with_retries(get_function, counter, key):
        @with_retries(counter=counter, key=key)
        def func(*args, **kvargs):
            rsp = get_function(*args, **kvargs)
            rsp.raise_for_status()
            rsp_size = len(rsp.content)
            LOG.debug('Got response of size %sbytes (%d bytes)', get_size_in_units(rsp_size), rsp_size)
            counter['collector.stats.total_response_bytes'] += rsp_size
            return rsp

        return func
