# coding=utf-8
from __future__ import unicode_literals

from datetime import date, datetime
from dateutil.relativedelta import relativedelta
from ticket_parser2.api.v1 import TvmClient, TvmClientStatus, TvmApiClientSettings

from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.lib_datetime import parse_datetime_iso, timestamp
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, SuburbanBoySuburbanOrderSnapshot


LOG = get_logger(__name__)


class SuburbanBoyCollector(HttpCollector):
    PARTNER_NAME = 'suburban'
    ORDER_TYPE = 'SUBURBAN'
    BASE_URL = 'http://api-prod.travel-hotels.yandex.net/api/cpa_export/v2/get_cpa_order_snapshots'
    ALLOW_PROXY = False
    REQUEST_TIMEOUT = 20

    STATUS_MAPPING = {
        'UNPAID': OrderStatus.UNPAID,
        'PAID': OrderStatus.PAID,
        'REFUNDED': OrderStatus.REFUNDED,
        'CONFIRMED': OrderStatus.CONFIRMED,
        'PENDING': OrderStatus.PENDING,
        'CANCELLED': OrderStatus.CANCELLED,
        'DELETED': OrderStatus.DELETED
    }

    def __init__(self, options):
        super(SuburbanBoyCollector, self).__init__(options)

        use_tvm = options.use_tvm
        tvm_ticket = None
        if use_tvm:
            tvm_secret = options.tvm_secret
            tvm_client_id = options.tvm_client_id
            tvm_service_id = options.tvm_service_id
            tvm_client = self.create_tvm_client(tvm_service_id, tvm_client_id, tvm_secret)
            tvm_ticket = tvm_client.get_service_ticket_for('cpa_export_api')
        LOG.debug('Got tvm ticket: {}'.format(tvm_ticket))
        self.tvm_ticket = tvm_ticket

        self.date_from = datetime.combine(parse_datetime_iso(options.date_from).date(), datetime.min.time())
        self.date_to = datetime.combine(parse_datetime_iso(options.date_to).date(), datetime.max.time())
        self.limit = options.limit

        self.get_raw_snapshots_with_limit = with_retries(
            self.get_raw_snapshots_once,
            counter=self.metrics,
            key='collector.events.invalid_response',
        )

    @staticmethod
    def create_tvm_client(tvm_service_id, tvm_client_id, tvm_secret):
        destinations = {'cpa_export_api': tvm_service_id}
        settings = TvmApiClientSettings(
            self_client_id=tvm_client_id,
            enable_service_ticket_checking=True,
            enable_user_ticket_checking=False,
            self_secret=tvm_secret,
            dsts=destinations,
        )
        client = TvmClient(settings)
        if client.status != TvmClientStatus.Ok:
            LOG.exception('Bad tvm client status: %s', client.status)
            raise ProcessError(ErrorType.ET_BAD_TVM_CLIENT_STATUS)
        return client

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('--use-tvm', action='store_true')
        parser.add_argument('--tvm-secret')
        parser.add_argument('--tvm-client-id', type=int)
        parser.add_argument('--tvm-service-id', type=int)

        parser.add_argument('--date-from', default=(date.today() - relativedelta(days=2)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())
        parser.add_argument('--limit', default=300)

    def get_raw_snapshots_once(self):
        params = {
            'order_type': self.ORDER_TYPE,
            'updated_at_from_utc': self.date_from.isoformat(),
            'updated_at_to_utc': self.date_to.isoformat(),
            'limit': self.limit,
        }
        headers = {}
        if self.tvm_ticket is not None:
            headers['X-Ya-Service-Ticket'] = self.tvm_ticket

        rsp = self.request_get(self.base_url, params=params, headers=headers)
        return rsp

    def get_raw_snapshots(self):
        has_more = True
        while has_more:
            rsp = self.get_raw_snapshots_with_limit()
            response = rsp.json()['order_snapshots']
            has_more = rsp.json()['has_more']
            if has_more:
                LOG.info('Will get one more page')
                self.date_from = parse_datetime_iso(response[-1]['updated_at'])
            yield response

    def get_order_snapshot(self, raw_snapshot):
        snapshot = SuburbanBoySuburbanOrderSnapshot()

        snapshot.update_partner_order_id(raw_snapshot['partner_order_id'])
        snapshot.boy_order_id = raw_snapshot['boy_order_id']
        snapshot.status = self.STATUS_MAPPING[raw_snapshot['order_status']]
        snapshot.label = raw_snapshot['label']
        snapshot.currency_code = OrderCurrencyCode(raw_snapshot['amount']['currency'])
        snapshot.order_amount = float(raw_snapshot['amount']['value'])
        snapshot.profit_amount = float(raw_snapshot['profit']['value'])
        snapshot.created_at = timestamp(parse_datetime_iso(raw_snapshot['created_at']))
        snapshot.updated_at = timestamp(parse_datetime_iso(raw_snapshot['updated_at']))

        # suburban specific
        snapshot.provider = raw_snapshot['provider']
        snapshot.provider_order_id = raw_snapshot['provider_order_id']
        snapshot.carrier_partner = raw_snapshot['carrier_partner']
        snapshot.station_from_id = raw_snapshot['station_from_id']
        snapshot.station_to_id = raw_snapshot['station_to_id']
        snapshot.station_from_title = raw_snapshot['station_from_title']
        snapshot.station_to_title = raw_snapshot['station_to_title']
        snapshot.departure_date = raw_snapshot['departure_date']

        LOG.debug(snapshot.as_dict())
        return snapshot

    def _get_snapshots(self):
        LOG.info('Getting order snapshots starting from %r to %r', self.date_from, self.date_to)
        for raw_snapshot_batch in self.get_raw_snapshots():
            for raw_snapshot in raw_snapshot_batch:
                yield self.get_order_snapshot(raw_snapshot)
