# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date, datetime, timedelta

from dateutil.relativedelta import relativedelta
from lxml import etree
from six import text_type

from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import iter_periods, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, LevelTravelWhitelabelOrderSnapshot


LOG = get_logger(__name__)


class LevelTravelWhitelabelCollector(HttpCollector):
    PARTNER_NAME = 'leveltravel_whitelabel'
    BASE_URL = 'https://level.travel/affiliate/conversions.xml'
    REQUEST_TIMEOUT = 90

    STATUS_MAPPING = {
        'PROCESSING': OrderStatus.PENDING,
        'PAID': OrderStatus.CONFIRMED,
        'CANCELED': OrderStatus.CANCELLED,
    }

    def __init__(self, options):
        super(LevelTravelWhitelabelCollector, self).__init__(options)
        self.auth = options.password
        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = (parse_datetime_iso(options.date_to).date()) + relativedelta(days=+1)
        self.single_report_days = int(options.single_report_days)
        self.get_report = with_retries(
            self.get_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)
        parser.add_argument('-p', '--password', required=True)
        parser.add_argument('--date-from', default=(date.today() - relativedelta(days=3)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())
        parser.add_argument('--single-report-days', default='14')

    def get_report_once(self, period_from, period_to):
        LOG.info('Requesting data for period [{}, {})'.format(period_from.isoformat(), period_to.isoformat()))
        params = {
            'partner': 'Ya',
            'password': self.auth,
            'start_date': period_from.isoformat(),
            'end_date': period_to.isoformat(),
        }
        rsp = self.request_get(self.base_url, timeout=self.REQUEST_TIMEOUT, params=params)
        return rsp.content

    def get_single_report_raw_orders(self, period_from, period_to):
        report_xml_content = self.get_report(period_from, period_to)
        try:
            report_xml_tree = etree.fromstring(report_xml_content, parser=etree.XMLParser(encoding='utf-8'))
        except etree.XMLSyntaxError:
            LOG.exception('Bad LevelTravel XML')
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        return report_xml_tree

    def get_all_raw_orders(self):
        for period_from, period_to in iter_periods(self.date_from, self.date_to, self.single_report_days):
            for order_xml_tree in self.get_single_report_raw_orders(period_from, period_to):
                yield order_xml_tree

    def _get_snapshots(self):
        for order_xml_tree in self.get_all_raw_orders():
            def parse_xml(element):
                if len(element) == 0:
                    return text_type(element.text)
                return {text_type(e.tag): parse_xml(e) for e in element}

            order_dict = parse_xml(order_xml_tree)
            partner_status = order_dict['state']

            snapshot = LevelTravelWhitelabelOrderSnapshot()
            snapshot.update_partner_order_id(order_dict['id'])
            snapshot.status = self.STATUS_MAPPING[partner_status]
            if not isinstance(order_dict.get('partner_fields'), unicode):  # strange case
                snapshot.label = order_dict.get('partner_fields', {}).get('label')
            snapshot.currency_code = OrderCurrencyCode(order_dict['currency'])
            snapshot.order_amount = float(order_dict['price'])
            snapshot.profit_amount = float(order_dict['profit'])
            snapshot.fuel_charge = float(order_dict['fuel_charge'])
            snapshot.created_at = timestamp(parse_datetime_iso(order_dict['created_at']))

            # tours specific
            snapshot.partner_status = partner_status

            if order_dict['extras'] is not None:
                parsed_checkin = datetime.strptime(order_dict['extras']['start_date'], '%d.%m.%Y').date()
                snapshot.check_in = text_type(datetime.strftime(parsed_checkin, '%Y-%m-%d'))
                snapshot.check_out = text_type(datetime.strftime(parsed_checkin + timedelta(int(order_dict['extras']['nights'])), '%Y-%m-%d'))
                snapshot.hotel_name = order_dict['extras']['hotel']
                snapshot.hotel_pansion = order_dict['extras']['pansion']

            snapshot.country_name = order_dict['country']
            snapshot.city_name = order_dict['city']
            snapshot.adults = int(order_dict['adults'])
            snapshot.children = int(order_dict['children'])
            snapshot.infants = int(order_dict['infants'])

            yield snapshot
