# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from collections import OrderedDict
from datetime import date, datetime

from dateutil.relativedelta import relativedelta
from six import text_type

from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import ImTrainOrderSnapshot, OrderCurrencyCode, OrderStatus, ItemState


LOG = get_logger(__name__)


IM_ORDER_FLOAT_FIELDS = [
    'order_amount',
    'profit_amount',
    'total_fee_amount',
    'total_insurance_amount',
    'total_insurance_profit_amount',
    'total_partner_fee_amount',
    'total_partner_refund_fee_amount',
    'total_refund_fee_amount',
    'total_refund_insurance_amount',
    'total_refund_ticket_amount',
    'total_service_amount',
    'total_tariff_amount'
]


class ImCollector(HttpCollector):
    PARTNER_NAME = 'im'
    BASE_URL = 'https://production.train-api.rasp.internal.yandex.net/ru/api/cpa'
    ALLOW_PROXY = False

    SET_HOST_HEADER = True
    REQUEST_TIMEOUT = 15

    STATUS_MAPPING = {
        'Cancelled': OrderStatus.CANCELLED,
        'Done': OrderStatus.CONFIRMED,
        'In progress': OrderStatus.PAID,
        'Reserved': OrderStatus.PENDING,
    }

    ITEM_STATE_MAPPING = {
        'Reserved': ItemState.RESERVED,
        'Done': ItemState.CONFIRMED,
        'Cancelled': ItemState.CANCELLED,
        'In progress': ItemState.PENDING,
    }

    def __init__(self, options):
        super(ImCollector, self).__init__(options)

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        self.get_day_orders = with_retries(
            self.get_day_orders_once,
            max_attempts=5,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)
        parser.add_argument('--date-from', default=(date.today() - relativedelta(days=2)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

    def _get_snapshots(self):
        dropped_snapshots_count = 0
        for day_date in iter_day(self.date_from, self.date_to):
            for snapshot in self.get_day_snapshots(day_date):
                if snapshot.partner_name != self.PARTNER_NAME:
                    LOG.warning('Unexpected partner name %r for uid = %r', snapshot.partner_name, snapshot.uid)
                    continue

                yield snapshot
        LOG.info('Dropped snapshots: %d', dropped_snapshots_count)

    def get_day_snapshots(self, day_date):
        LOG.info('Processing {}'.format(day_date))

        snapshots = OrderedDict()

        for row in self.get_day_orders(day_date, 'orders'):
            key = row['uid']
            snapshots[key] = row

        for row in self.get_day_orders(day_date, 'orders-with-refunds'):
            key = row['uid']
            snapshots[key] = row

        for raw_snapshot in snapshots.values():
            yield self.get_order_snapshot(raw_snapshot)

    def get_day_orders_once(self, day_date, relative_path):
        day_time = datetime(year=day_date.year, month=day_date.month, day=day_date.day)
        url = '{}/{}/'.format(self.base_url, relative_path)

        rsp = self.request_get(
            url,
            timeout=self.REQUEST_TIMEOUT,
            params={
                'utc_from': day_time,
                'utc_to': day_time + relativedelta(days=1),
                'format': 'json',
                'flat': False,
            },
        )

        rsp_json = rsp.json()

        for order in rsp_json['results']['orders']:
            yield order

        next_page = rsp_json['next']

        while next_page:
            rsp = self.request_get(next_page)
            rsp_json = rsp.json()

            for order in rsp_json['results']['orders']:
                yield order

            next_page = rsp_json['next']

    def get_order_snapshot(self, raw_snapshot):
        partner_status = raw_snapshot['travel_status']
        raw_snapshot['train_internal_status'] = raw_snapshot['status']
        raw_snapshot['status'] = self.STATUS_MAPPING[partner_status]
        raw_snapshot['item_state'] = self.ITEM_STATE_MAPPING[partner_status]
        raw_snapshot['boy_order_id'] = raw_snapshot['uid']

        for field in IM_ORDER_FLOAT_FIELDS:
            ImCollector.fix_float(raw_snapshot, field)

        snapshot = ImTrainOrderSnapshot.from_dict(raw_snapshot, ignore_unknown=True)

        snapshot.update_partner_order_id(text_type(raw_snapshot['partner_data_im_order_id']))
        snapshot.currency_code = OrderCurrencyCode.RUB
        snapshot.active_ticket_count = snapshot.total_ticket_count - snapshot.refunded_ticket_count

        return snapshot

    @staticmethod
    def fix_float(raw_snapshot, field_name):
        raw_snapshot[field_name] = float(raw_snapshot.get(field_name, .0))
