# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from argparse import ArgumentParser
import json
import logging.config

from travel.cpa.collectors.nomenclature import COLLECTORS
from travel.cpa.lib.common import hide_secrets
from travel.cpa.lib.lb_writer import LBWriter
from travel.cpa.lib.lib_logging import LOG_CONFIG, get_logger
from travel.cpa.lib.metered_task_executor import MeteredTaskExecutor
from travel.library.python.tools import replace_args_from_env


LOG = get_logger('collectors.main')


class Executor(MeteredTaskExecutor):

    def __init__(self, collector_cls, options, common_labels):
        super(Executor, self).__init__('collector.stats', options, common_labels)
        self.collector_cls = collector_cls
        self.options = options

    def try_execute(self):
        collector = self.collector_cls(self.options)
        snapshots = collector.get_snapshots()
        if self.options.dump_snapshots:
            snapshots = self.dump_snapshots(snapshots, self.collector_cls.PARTNER_NAME + '.jsonl')

        if self.options.lb_skip_send:
            self.dummy_read(snapshots)
        else:
            self.send_data_to_lb(snapshots)

        LOG.info('Collector metrics: %r', collector.metrics)
        self.send_dict_metrics(self.solomon_client, collector.metrics)

    @staticmethod
    def dump_snapshots(snapshots, fn):
        with open(fn, 'w') as f:
            for snapshot in snapshots:
                yield snapshot
                line = json.dumps(snapshot.as_dict()) + '\n'
                f.write(line)

    @staticmethod
    def convert_snapshots(snapshots):
        for snapshot in snapshots:
            yield snapshot.as_dict()

    def send_data_to_lb(self, snapshots):
        source_id = b'cpa-{}-{}'.format(self.collector_cls.PARTNER_NAME, self.options.lb_source_id)
        if self.options.low_priority_order_processing:
            source_id += b'-low-priority'
        lb_token = self.options.lb_token
        if lb_token is not None:
            lb_token = str(lb_token)
        writer = LBWriter(self.options.lb_url, self.options.lb_port, source_id, lb_token)
        topic = self.options.lb_topic
        writer.write(topic, self.convert_snapshots(snapshots))
        LOG.info('Data sent to LB')

    @staticmethod
    def dummy_read(snapshots):
        for snapshot in snapshots:
            snapshot.validate()


def main():
    init_parser = ArgumentParser()
    collector_map = {c.PARTNER_NAME: c for c in COLLECTORS}
    init_parser.add_argument('--partner-name', required=True, choices=collector_map.keys())
    init_options, rest_args = init_parser.parse_known_args()
    partner_name = init_options.partner_name
    collector_cls = collector_map[partner_name]

    parser = ArgumentParser()
    parser.add_argument('--lb-skip-send', action='store_true')
    parser.add_argument('--lb-url', default='logbroker.yandex.net')
    parser.add_argument('--lb-port', type=int, default=2135)
    parser.add_argument('--lb-token', default=None)
    parser.add_argument('--lb-topic', default=b'travel-cpa@dev--collectors-result')
    parser.add_argument('--lb-source-id', default='0')
    parser.add_argument('--low-priority-order-processing', action='store_true')

    parser.add_argument('--dump-snapshots', action='store_true')

    parser.add_argument(
        '--log-level',
        action='append',
        help='overrides default log level for module, format: module_name=LEVEL'
    )

    collector_cls.configure_all(parser)
    collector_cls.configure(parser)
    Executor.configure(parser)
    options = parser.parse_args(replace_args_from_env(rest_args))

    loggers = LOG_CONFIG['loggers']
    log_level = options.log_level or []
    for l in log_level:
        logger, level = l.split('=', 1)
        loggers[logger] = {'level': level}
    logging.config.dictConfig(LOG_CONFIG)

    LOG.info('Collector configured')

    displayed_options = hide_secrets(vars(options))
    LOG.info('Working with: %r', displayed_options)

    common_labels = {'partner': collector_cls.PARTNER_NAME}
    Executor.resolve_secrets(options, common_labels)
    Executor(collector_cls, options, common_labels).execute()


if __name__ == '__main__':
    main()
