# -*- coding: utf-8 -*-

from base64 import b64decode
from datetime import datetime
from email import message_from_bytes
from io import BytesIO
from parse import compile
from typing import IO, List, NamedTuple
import imaplib
import logging
import os

from travel.cpa.data_check.reconciliation.datetime_tools import parse_datetime_email
from travel.library.python.tools import with_retries
from travel.library.python.imap.oauth_tools import imap_oauth_authenticate


LOG = logging.getLogger(__name__)

UID_PATTERN = compile('(UID {uid})')


class Attachment(NamedTuple):
    file_name: str
    data: IO


class Message(NamedTuple):
    uid: str
    datetime: datetime
    subject: str
    attachments: List[Attachment]
    sender: str


class EmailClient(object):

    def __init__(self, login, imap_token, debug_mode):
        self.client = None
        self.debug_mode = debug_mode
        if debug_mode:
            # using a local report file (if any) instead of the real mail server
            self.get_new_reports = self.get_new_reports_debug
        else:
            self.do_login(login, imap_token)

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if not self.debug_mode:
            self.client.logout()

    @staticmethod
    def parse_uid(data):
        match = UID_PATTERN.search(data)
        return match['uid']

    @with_retries
    def do_login(self, login, imap_token):
        client = imaplib.IMAP4_SSL('imap.yandex-team.ru')
        imap_oauth_authenticate(client, login, imap_token)
        self.client = client

    def get_new_reports(self, input_mailbox, processed_mailbox, use_flags_instead_of_move):
        if use_flags_instead_of_move and processed_mailbox is not None:
            raise Exception('use_flags_instead_of_move can\'t be used with processed_mailbox')
        if not use_flags_instead_of_move and processed_mailbox is None:
            raise Exception('Either use_flags_instead_of_move should be true or processed_mailbox should be set')
        client = self.client
        result = client.select(input_mailbox, readonly=False)
        if result[0] != 'OK':
            raise Exception('Failed to select {} mailbox: {}'.format(input_mailbox, result[1]))
        LOG.info('Working with mailbox %s', input_mailbox)

        messages = list()
        _, message_ids = client.search(None, 'UNFLAGGED' if use_flags_instead_of_move else 'ALL')
        message_ids = message_ids[0].split()
        LOG.info('Got %d messages', len(message_ids))
        for message_id in message_ids:
            _, data = client.fetch(message_id, 'UID')
            LOG.debug('UID = "%s"', data)
            message_uid = self.parse_uid(data[0].decode())
            LOG.debug('message_uid = "%s"', message_uid)

            _, data = client.fetch(message_id, '(RFC822)')
            msg = message_from_bytes(data[0][1])
            attachments = list()
            for part in msg.walk():
                content_disposition = part.get('Content-Disposition')
                if content_disposition is not None and content_disposition.startswith('attachment;'):
                    fn = part.get_filename()
                    if fn.startswith('=?'):
                        LOG.debug(fn)
                        _, encoding, _, fn, *_ = fn.split('?')
                        fn = b64decode(fn).decode(encoding)
                    content = part.get_payload(decode=True)
                    f = BytesIO(content)
                    attachments.append(Attachment(file_name=fn, data=f))
            message = Message(
                uid=message_uid,
                datetime=parse_datetime_email(msg['Date']) or datetime.now(),
                subject=msg['Subject'],
                attachments=attachments,
                sender=msg['From'],
            )
            messages.append(message)

        for message in messages:
            yield message
            if use_flags_instead_of_move:
                flag = '\\Flagged'
                LOG.debug(f'Add "{flag}" flag to message')
                result = client.store(message.uid, '+FLAGS', f'({flag})')
                if result[0] != 'OK':
                    raise Exception(f'Failed to mark message with flag "{flag}", {result}')
                LOG.info(f'Message marked with flag "{flag}"')
            else:
                LOG.debug('Move message to %s', processed_mailbox)
                result = client.uid('MOVE', message.uid, processed_mailbox)
                if result[0] != 'OK':
                    raise Exception('Failed to move message, {}'.format(result))
                LOG.info('Message moved to processed')
        client.expunge()
        client.close()

    # debug function
    @staticmethod
    def get_new_reports_debug(mailbox, processed_mailbox):
        fn = mailbox + '.xlsx'
        if not os.path.exists(fn):
            yield Message(
                uid='0',
                datetime=datetime.now(),
                subject='2019-02',
                attachments=list(),
                sender=''
            )
            return

        LOG.info('Debug mode is ON. Reading local report file %s', fn)
        with open(fn, 'rb') as f:
            yield Message(
                uid='0',
                datetime=datetime.now(),
                subject='2019-09',
                attachments=[Attachment(file_name=fn, data=f)],
                sender=''
            )
