# -*- coding: utf-8 -*-
import re
from datetime import datetime, timedelta
import logging
import os

from dateutil.relativedelta import relativedelta
from dateutil.tz import gettz, tzlocal
from typing import Dict, Optional
from yt.wrapper.client import YtClient

from travel.cpa.data_check.reconciliation.datetime_tools import parse_date_iso, DateRange, get_month_range
from travel.cpa.data_check.reconciliation.diff_info import OrderDiff, ReportDiff
from travel.cpa.data_check.reconciliation.errors import ErrorType, ProcessError
from travel.cpa.data_check.reconciliation.partner_report import CsvReport, XlsxReport, ExpediaReport
from travel.cpa.data_check.reconciliation.partner_report import OrderInfo
from travel.cpa.data_check.reconciliation.reconciliation_report import ReconciliationReport
from travel.cpa.data_check.reconciliation.st_client import IssueInfo

from travel.library.python.currency_converter import CurrencyConverter

LOG = logging.getLogger(__name__)


class Partner(object):

    __input_mailbox__ = None
    __use_flags_instead_of_move__ = False
    __partner_name__ = None
    __month_offset__ = None
    __report_builder__ = None
    __report_fields__ = None
    __date_field__ = None
    __profit_amount_field__ = None
    __timezone__ = None
    __check_order_amount__ = True
    __check_profit_amount__ = True
    __relative_comparison_threshold__ = None
    __need_currency_converter__ = False

    def __init__(self, email_client, st_client, options):
        self.email_client = email_client
        self.st_client = st_client
        self.options = options
        self.time_delta = relativedelta()
        if self.__need_currency_converter__:
            self.currency_converter: Optional[CurrencyConverter] = CurrencyConverter(base_url=options.stocks_url)
        else:
            self.currency_converter: Optional[CurrencyConverter] = None
        if self.__timezone__ is not None:
            utc_now = datetime.utcnow()
            self.time_delta = gettz(self.__timezone__).utcoffset(utc_now) - tzlocal().utcoffset(utc_now)

    @property
    def name(self):
        return self.__partner_name__

    @property
    def report_fields(self):
        return self.__report_fields__

    @staticmethod
    def price_fixer(value):
        return value

    def should_skip_cpa_row(self, row):
        return False

    def patch_cpa_row(self, row):
        return row

    def get_report_dates_range(self, day_date, subject) -> DateRange:
        """
        calculates dates range of partner report
        1) if :subject: has format of yyyy-mm => take range of corresponding month
        2) else => month of :day_date: - __month_offset__ (range of that month)
        """
        try:
            return get_month_range(parse_date_iso(subject + '-01'))
        except Exception:
            pass
        return get_month_range(day_date + relativedelta(months=-self.__month_offset__, day=1))

    @staticmethod
    def try_convert_field(d, field, converter):
        field_value = d.get(field)
        if field_value is not None:
            d[field] = converter(field_value)

    def time_converter(self, ts):
        return datetime.fromtimestamp(ts) + self.time_delta

    def get_cpa_data(self, orders_table, partner_name, dates_range, date_field, profit_amount_field) -> Dict[str, OrderInfo]:
        cpa_orders = dict()
        cpa_all_orders = dict()
        yt_client = YtClient(proxy='hahn', token=self.options.yt_token)
        for row in yt_client.read_table(orders_table, enable_read_parallel=True):
            self.try_convert_field(row, 'created_at', self.time_converter)
            self.try_convert_field(row, 'updated_at', self.time_converter)
            self.try_convert_field(row, 'check_in', parse_date_iso)
            self.try_convert_field(row, 'check_out', parse_date_iso)
            self.try_convert_field(row, 'is_suspicious', str)
            self.try_convert_field(row, 'has_label', str)
            if self.should_skip_cpa_row(row):
                continue
            row = self.patch_cpa_row(row)
            row_partner_name = row['partner_name']
            order_id = str(row['partner_order_id'])
            cpa_all_orders[order_id] = row
            if row_partner_name != partner_name or not (dates_range.from_date <= row[date_field] <= dates_range.to_date):
                continue
            order_info = OrderInfo(
                order_id=order_id,
                status=row['status'],
                order_amount=row['order_amount'],
                profit_amount=row[profit_amount_field],
                profit_amount_rub=row['profit_amount_rub'],
                currency_code=row['currency_code'],
                field_values=row,
            )
            cpa_orders[order_id] = order_info
        return cpa_orders

    def get_report_diff(self, report_dates: DateRange, partner_orders: Dict[str, OrderInfo], cpa_orders: Dict[str, OrderInfo]):
        partner_keys = set(partner_orders.keys())
        cpa_keys = set(cpa_orders.keys())

        partner_extra_keys = partner_keys - cpa_keys
        cpa_extra_keys = cpa_keys - partner_keys
        common_keys = partner_keys & cpa_keys

        differed_orders = list()
        for key in common_keys:
            partner_order = partner_orders[key]
            cpa_order = cpa_orders[key]

            if cpa_order.field_values['status'] in ('refunded', 'cancelled'):
                continue

            has_diff = False
            fail_reason = list()

            exchange_rate = 1
            if self.__need_currency_converter__:
                if partner_order.currency_code != cpa_order.currency_code:
                    if cpa_order.currency_code != 'RUB':
                        msg = f'Can\'t process non-rub order ({cpa_order.currency_code=}, {cpa_order.order_id=})'
                        raise Exception(msg)
                    exchange_rate = self.currency_converter.get_rate(
                        partner_order.currency_code, cpa_order.field_values['created_at'].date()
                    )

            partner_value = float(partner_order.order_amount) * exchange_rate
            cpa_value = cpa_order.order_amount
            order_amount_diff, is_significant_diff = self._compare_prices(partner_value, cpa_value)
            if self.__check_order_amount__ and is_significant_diff:
                has_diff = True
                fail_reason.append('order amount differs')

            profit_amount_diff = None
            if self.__check_profit_amount__:
                try:
                    partner_value = float(partner_order.profit_amount) * exchange_rate
                except ValueError:
                    LOG.warning(f'Failed to convert partner profit_amount to float {partner_order}')
                    partner_value = 0
                cpa_value = cpa_order.profit_amount
                profit_amount_diff, is_significant_diff = self._compare_prices(partner_value, cpa_value)
                if is_significant_diff:
                    has_diff = True
                    fail_reason.append('profit amount differs')
                if cpa_order.status == 'confirmed' and cpa_order.profit_amount_rub < 10:
                    has_diff = True
                    fail_reason.append('low profit')

            if has_diff:
                order_diff = OrderDiff(
                    order_id=key,
                    order_amount_diff=order_amount_diff,
                    profit_amount_diff=profit_amount_diff,
                    fail_reason=', '.join(fail_reason),
                )
                differed_orders.append(order_diff)

        report_diff = ReportDiff(
            report_dates=report_dates,
            partner_extra_orders=list(partner_extra_keys),
            cpa_extra_orders=list(cpa_extra_keys),
            differed_orders=differed_orders,
        )
        return report_diff

    def _compare_prices(self, partner_value, cpa_value) -> (float, bool):
        diff_value = abs(partner_value - cpa_value)
        if self.__relative_comparison_threshold__ is None:
            significant_diff = diff_value >= 0.01
        else:
            significant_diff = 1.0 * diff_value / max(1, min(partner_value, cpa_value)) >= self.__relative_comparison_threshold__
        return diff_value, significant_diff

    def get_issue_info(self, report_diff: ReportDiff, assignee, attachments, related_issue):
        queue = 'HOTELS'
        dates = report_diff.report_dates
        if dates.is_full_month():
            formatted_dates = dates.from_date.strftime('%Y-%m')
        else:
            formatted_dates = '{} - {}'.format(dates.from_date.strftime('%Y-%m-%d'), dates.to_date.strftime('%Y-%m-%d'))
        summary = 'Отчёт по сверке, {}, {}'.format(self.__partner_name__, formatted_dates)

        if not (report_diff.differed_orders or report_diff.partner_extra_orders or report_diff.cpa_extra_orders):
            issue_info = IssueInfo(
                queue=queue,
                summary=summary,
                description='Всё сходится!!!',
                assignee=assignee,
                status='fixed',
                attachments=list(),
                related_issue=related_issue,
            )
            return issue_info

        description = list()

        if report_diff.partner_extra_orders:
            line = 'Нет в CPA (количество заказов): **{}**'.format(len(report_diff.partner_extra_orders))
            description.append(line)

        if report_diff.cpa_extra_orders:
            line = 'Нет в отчёте партнёра (количество заказов): **{}**'.format(len(report_diff.cpa_extra_orders))
            description.append(line)

        if report_diff.differed_orders:
            line = 'Отличающиеся заказы (количество): **{}**'.format(len(report_diff.differed_orders))
            description.append(line)

        description = '\n'.join(description)

        issue_info = IssueInfo(
            queue=queue,
            summary=summary,
            description=description,
            assignee=assignee,
            status='open',
            attachments=attachments,
            related_issue=related_issue,
        )
        return issue_info

    def process(self):
        LOG.info('Processing %s', self.__partner_name__)
        if self.__input_mailbox__ is None:
            input_mailbox = 'travel-hotels-partner-{}'.format(self.__partner_name__)
        else:
            input_mailbox = self.__input_mailbox__
        processed_mailbox = None if self.__use_flags_instead_of_move__ else 'processed-{}'.format(self.__partner_name__)
        messages = self.email_client.get_new_reports(input_mailbox, processed_mailbox, self.__use_flags_instead_of_move__)
        for message in messages:
            try:
                if self.should_process_message(message):
                    self.try_process_message(message)
            except ProcessError as e:
                LOG.exception(e.error_type.value)
                self.send_error_message(e.error_type, message.subject)

    def should_process_message(self, message):
        return True

    def try_process_message(self, message):
        LOG.info('Processing message')
        attachment_count = len(message.attachments)
        LOG.info('Got {} attachments'.format(attachment_count))
        if attachment_count == 0:
            raise ProcessError(ErrorType.ET_NO_ATTACHMENTS)
        if attachment_count > 1:
            raise ProcessError(ErrorType.ET_MANY_ATTACHMENTS)

        raw_report = message.attachments[0]
        partner_orders: Dict[str, OrderInfo] = self.__report_builder__.build(fn=raw_report.data)
        LOG.info('Partner report order count: %s', len(partner_orders))

        LOG.debug('Message datetime: %r', message.datetime)
        LOG.debug('Message subject: %s', message.subject)
        report_dates = self.get_report_dates_range(message.datetime.date(), message.subject)
        LOG.info('Checking report of %r - %r', report_dates.from_date, report_dates.to_date)
        cpa_orders = self.get_cpa_data(
            orders_table=self.options.orders_table,
            partner_name=self.__partner_name__,
            dates_range=report_dates,
            date_field=self.__date_field__,
            profit_amount_field=self.__profit_amount_field__,
        )
        LOG.info('CPA order count: %s', len(cpa_orders))

        report_diff = self.get_report_diff(report_dates, partner_orders, cpa_orders)

        ext = os.path.splitext(raw_report.file_name)[1]
        partner_report_fn = '{}_partner_report{}'.format(self.__partner_name__, ext)
        raw_report.data.seek(0)
        report_data = raw_report.data.read()
        with open(partner_report_fn, 'wb') as f:
            f.write(report_data)

        with ReconciliationReport(self, self.options.debug_mode) as reconciliation_report:
            LOG.info('Writing reconciliation report')
            reconciliation_report.make_report(report_diff, partner_orders, cpa_orders)
            issue_info = self.get_issue_info(
                report_diff=report_diff,
                assignee=self.options.assignee,
                attachments=[partner_report_fn, reconciliation_report.report_file_name],
                related_issue=self.options.related_issue,
            )
            LOG.info('Creating st issue')
            self.st_client.create_issue(issue_info)

        if os.path.exists(partner_report_fn):
            os.remove(partner_report_fn)

        LOG.info('Message processing finished')

    def send_error_message(self, error_type, message_subject):
        queue = 'HOTELS'
        summary = 'Отчёт по сверке, {}'.format(self.__partner_name__)
        description = error_type.value
        if error_type == ErrorType.ET_REPORT_FORMAT:
            description = 'Неподходящий формат файла. Ожидается: {}'.format(self.__report_builder__.get_format_name())
        description += f'\nТема письма: <[{message_subject}]>'

        issue_info = IssueInfo(
            queue=queue,
            summary=summary,
            description=description,
            assignee=self.options.assignee,
            status='open',
            attachments=list(),
            related_issue=self.options.related_issue,
        )
        self.st_client.create_issue(issue_info)


class Booking(Partner):

    __partner_name__ = 'booking'
    __month_offset__ = 2
    __report_builder__ = CsvReport(order_id_col_name='hotelreservation_id',
                                   order_amount_col_name='amount',
                                   profit_amount_col_name='partner_amount')
    __report_fields__ = [
        ('partner_order_id', 'hotelreservation_id'),
        ('status', None),
        ('partner_status', 'status'),
        ('order_amount', None),
        ('order_amount_rub', 'amount'),
        ('profit_amount', 'partner_amount'),
        (None, 'exchange_rate'),
        ('created_at', 'booked'),
        ('updated_at', None),
        ('check_in', 'checkin'),
        ('check_out', 'checkout'),
        ('hotel_name', 'hotel_name'),
        ('hotel_country', None),
        ('hotel_city', None),
        ('is_suspicious', None),
        ('has_label', None),
    ]

    __date_field__ = 'check_out'
    __profit_amount_field__ = 'profit_amount'
    __check_order_amount__ = False


class Hotels101(Partner):

    __partner_name__ = 'hotels101'
    __month_offset__ = 2
    __report_builder__ = XlsxReport(order_id_col_name='ID брони',
                                    order_amount_col_name='Сумма брони',
                                    profit_amount_col_name='Комиссия')
    __report_fields__ = [
        ('partner_order_id', 'ID брони'),
        ('status', None),
        ('partner_status', 'Статус'),
        ('order_amount', 'Сумма брони'),
        ('profit_amount', 'Комиссия'),
        ('created_at', 'Время/Дата'),
        ('updated_at', None),
        ('check_in', 'Дата заезда'),
        ('check_out', None),
        ('hotel_name', 'Отель'),
        ('hotel_country', 'Страна'),
        ('hotel_city', 'Регион'),
        ('is_suspicious', None),
        ('has_label', None),
    ]

    __date_field__ = 'check_out'
    __profit_amount_field__ = 'profit_amount'


class Hotelscombined(Partner):

    __partner_name__ = 'hotelscombined'
    __month_offset__ = 1
    __report_builder__ = XlsxReport(order_id_col_name='Conversion Booking ID',
                                    order_amount_col_name='Revenue',
                                    profit_amount_col_name='Affiliate Commission',
                                    sheet_name='Booking Details')
    __report_fields__ = [
        ('partner_order_id', 'Conversion Booking ID'),
        ('status', None),
        ('partner_status', 'Status'),
        ('order_amount', 'Revenue'),
        ('profit_amount', 'Affiliate Commission'),
        ('created_at', 'Booking Date'),
        ('updated_at', None),
        ('check_in', 'Checkin'),
        ('check_out', 'Checkout'),
        ('hotel_name', 'Hotel Name'),
        ('hotel_country', 'Hotel Country'),
        ('hotel_city', 'Hotel City'),
        ('is_suspicious', None),
        ('has_label', None),
    ]

    __date_field__ = 'check_out'
    __profit_amount_field__ = 'profit_amount'
    __timezone__ = 'UTC+11'


class Ostrovok(Partner):

    __partner_name__ = 'ostrovok'
    __month_offset__ = 2
    __report_builder__ = XlsxReport(order_id_col_name='Номер заказа',
                                    order_amount_col_name='Стоимость бронирования',
                                    profit_amount_col_name='Коммиссия без НДС')
    __report_fields__ = [
        ('partner_order_id', 'Номер заказа'),
        ('status', None),
        ('partner_status', None),
        ('order_amount', 'Стоимость бронирования'),
        ('profit_amount', 'Коммиссия без НДС'),
        ('created_at', 'Дата создания'),
        ('updated_at', None),
        ('check_in', 'Заезд'),
        ('check_out', 'Выезд'),
        ('hotel_name', 'Отель'),
        ('hotel_country', None),
        ('hotel_city', None),
        ('is_suspicious', None),
        ('has_label', None),
    ]

    __date_field__ = 'check_out'
    __profit_amount_field__ = 'profit_amount_ex_tax'


class Tvil(Partner):

    __partner_name__ = 'tvil'
    __month_offset__ = 1
    __report_builder__ = CsvReport(order_id_col_name='id',
                                   order_amount_col_name='amount',
                                   profit_amount_col_name=None,
                                   price_fixer=lambda x: Tvil.price_fixer(x))
    __report_fields__ = [
        ('partner_order_id', 'id'),
        ('status', None),
        ('partner_status', 'complete_cause'),
        ('order_amount', 'amount'),
        ('profit_amount', None),
        ('created_at', 'create_time'),
        ('updated_at', None),
        ('check_in', 'arrival_date'),
        ('check_out', 'departure_date'),
        ('hotel_name', None),
        ('hotel_country', None),
        ('hotel_city', None),
        ('is_suspicious', None),
        ('has_label', None),
    ]

    __date_field__ = 'check_out'
    __profit_amount_field__ = 'profit_amount_ex_tax'
    __check_profit_amount__ = False

    @staticmethod
    def price_fixer(value):
        return value.split()[0].replace(',', '.')


class Expedia(Partner):

    __input_mailbox__ = 'Yandex|travel-hotels-partner-expedia'
    __use_flags_instead_of_move__ = True
    __partner_name__ = 'expedia'
    __report_builder__ = ExpediaReport(sheet_name=None,
                                       order_id_col_name='Affiliate Reference Number',
                                       order_amount_col_name='Amount',
                                       profit_amount_col_name='Marketing Fee Amount',
                                       price_fixer=lambda x: Expedia.price_fixer(x),
                                       skip_head_rows=12,
                                       skip_tail_rows=2)
    __report_fields__ = [
        ('partner_order_id', 'Affiliate Reference Number'),
        ('status', None),
        ('partner_status', None),
        ('order_amount', 'Amount'),
        ('profit_amount', 'Marketing Fee Amount'),
        ('created_at', None),
        ('updated_at', None),
        ('check_in', 'Use Date Begin '),
        ('check_out', 'Use Date End'),
        ('hotel_name', 'Hotel Name'),
        ('hotel_country', 'Hotel Country'),
        ('hotel_city', 'Hotel City'),
        ('is_suspicious', None),
        ('has_label', None),
        ('currency_code', 'Currency Code'),
    ]

    __date_field__ = 'check_out'
    __profit_amount_field__ = 'profit_amount'
    __check_profit_amount__ = False
    __relative_comparison_threshold__ = 0.05
    __need_currency_converter__ = True

    def should_process_message(self, message):
        return 'Transaction Statement' in message.subject and 'startrek@yandex-team.ru' not in message.sender

    def get_report_dates_range(self, day_date, subject) -> DateRange:
        match = re.match('Yandex LLC Transaction Statement \\d+ For Period Ending (\\d{2}-\\w{3}-\\d{4})', subject.replace('\r\n', ''))
        if match is None:
            raise ProcessError(ErrorType.ET_INVALID_SUBJECT)
        subj_date = datetime.strptime(match.group(1), '%d-%b-%Y').date()
        return DateRange(from_date=(subj_date - timedelta(14)), to_date=(subj_date - timedelta(1)))  # 14-days range

    def should_skip_cpa_row(self, row):
        return row['status'] == 'cancelled'

    def patch_cpa_row(self, row):
        if row['partner_order_id'].endswith(':0'):
            row['partner_order_id'] = row['partner_order_id'][:-2]
        return row

    @staticmethod
    def price_fixer(value):
        if isinstance(value, str):
            if value.startswith('(') and value.endswith(')'):
                return float(value[1:-1]) * -1
        return value

PARTNERS = [
    Booking,
    Hotels101,
    Hotelscombined,
    Ostrovok,
    Tvil,
    Expedia,
]
