# -*- coding: utf-8 -*-

from codecs import BOM_UTF8
from datetime import datetime
from io import StringIO
from zipfile import BadZipfile
import csv
from typing import Any, Dict, NamedTuple, List

from openpyxl.utils import get_column_letter
import openpyxl
import xlrd

from travel.cpa.data_check.reconciliation.errors import ErrorType, ProcessError


class OrderInfo(NamedTuple):
    order_id: str
    status: str
    order_amount: float
    profit_amount: float
    profit_amount_rub: float
    currency_code: str
    field_values: Dict[str, Dict[str, Any]]


class CsvReport(object):
    def __init__(self, order_id_col_name, order_amount_col_name, profit_amount_col_name, price_fixer=None):
        self.order_id_col_name = order_id_col_name
        self.order_amount_col_name = order_amount_col_name
        self.profit_amount_col_name = profit_amount_col_name
        self.price_fixer = price_fixer or (lambda x: x)

    def build(self, fn):
        orders = dict()
        if hasattr(fn, 'read'):
            return self.get_fp_orders(fn, orders)
        with open(fn) as f:
            return self.get_fp_orders(f, orders)

    def get_format_name(self):
        return 'CSV'

    def get_fp_orders(self, fp, orders):
        try:
            return self.try_get_fp_orders(fp, orders)
        except csv.Error:
            raise ProcessError(ErrorType.ET_REPORT_FORMAT)

    def try_get_fp_orders(self, fp, orders):
        message_bytes = fp.read()
        if message_bytes.startswith(BOM_UTF8):
            message_bytes = message_bytes[len(BOM_UTF8):]
        fp = StringIO(message_bytes.decode('utf8'))

        reader = csv.DictReader(fp, restkey='others')
        for row in reader:
            order_info = OrderInfo(
                order_id=row[self.order_id_col_name],
                status='',
                order_amount=self.price_fixer(row[self.order_amount_col_name]),
                profit_amount=self.price_fixer(row[self.profit_amount_col_name]) if self.profit_amount_col_name else None,
                profit_amount_rub=0,
                currency_code='RUB',
                field_values=dict(row),
            )
            orders[order_info.order_id] = order_info
        return orders


class XlsxReport(object):
    def __init__(self, order_id_col_name, order_amount_col_name, profit_amount_col_name, sheet_name=None, price_fixer=None):
        self.order_id_col_name = order_id_col_name
        self.order_amount_col_name = order_amount_col_name
        self.profit_amount_col_name = profit_amount_col_name
        self.sheet_name = sheet_name
        self.price_fixer = price_fixer or (lambda x: x)

    def build(self, fn):
        try:
            return self.try_get_orders(fn)
        except BadZipfile:
            raise ProcessError(ErrorType.ET_REPORT_FORMAT)

    def get_format_name(self):
        return 'XLSX'

    @staticmethod
    def get_col_range(ws, col_letter):
        return '{0}{1}:{0}{2}'.format(col_letter, ws.min_row, ws.max_row)

    @staticmethod
    def get_row_range(ws, row_num):
        min_letter = get_column_letter(1)
        max_letter = get_column_letter(ws.max_column)
        return '{0}{1}:{2}{1}'.format(min_letter, row_num, max_letter)

    @staticmethod
    def get_row_values(ws, row_num):
        row_range = XlsxReport.get_row_range(ws, row_num)
        for cell in list(ws[row_range])[0]:
            yield cell.value

    def try_get_orders(self, fn):
        wb = openpyxl.load_workbook(fn)
        if self.sheet_name is None:
            ws = wb.active
        else:
            ws = wb[self.sheet_name]
        columns = list(self.get_row_values(ws, 1))
        orders = dict()
        for row_num in range(2, ws.max_row + 1):
            values = self.get_row_values(ws, row_num)
            order = dict(zip(columns, values))
            order_id = order[self.order_id_col_name]
            # dirty hack for hotels101 report
            if isinstance(order_id, float):
                order_id = int(order_id)
            order_id = str(order_id)
            # end of dirty hack
            order_info = OrderInfo(
                order_id=order_id,
                status='',
                order_amount=self.price_fixer(order[self.order_amount_col_name]),
                profit_amount=self.price_fixer(order[self.profit_amount_col_name]),
                profit_amount_rub=0,
                currency_code='RUB',
                field_values=order,
            )
            orders[order_id] = order_info
        return orders


class ExpediaReport:
    def __init__(self, sheet_name, order_id_col_name, order_amount_col_name, profit_amount_col_name, price_fixer, skip_head_rows=0, skip_tail_rows=0):
        self.sheet_name = sheet_name
        self.order_id_col_name = order_id_col_name
        self.order_amount_col_name = order_amount_col_name
        self.profit_amount_col_name = profit_amount_col_name
        self.price_fixer = price_fixer
        self.skip_head_rows = skip_head_rows
        self.skip_tail_rows = skip_tail_rows

    def build(self, fn):
        try:
            return self.try_get_orders(fn)
        except BadZipfile:
            raise ProcessError(ErrorType.ET_REPORT_FORMAT)

    def get_format_name(self):
        return 'XLS'

    def try_get_orders(self, fn):
        wb = xlrd.open_workbook(file_contents=fn.read())
        if self.sheet_name is None:
            ws = wb.sheet_by_index(0)
        else:
            ws = wb.sheet_by_name(self.sheet_name)

        columns = [x.value for x in ws.row(self.skip_head_rows)]
        rows = []
        for row_ind in range(1 + self.skip_head_rows, ws.nrows - self.skip_tail_rows):
            values = [x.value for x in ws.row(row_ind)]
            rows.append(dict(zip(columns, values)))
        return self.convert_rows_to_orders(rows)

    def convert_rows_to_orders(self, rows: List[Dict]) -> Dict[str, OrderInfo]:
        cols_blacklist = ['Transaction Type', 'Transaction Date', 'Amount', 'Marketing Fee Amount', 'Amount Payable', 'EAC Oracle Ref.']
        orders = dict()
        for row in rows:
            for name in ['Use Date Begin ', 'Use Date End']:
                row[name] = datetime.strptime(row[name], '%d-%b-%Y').date()
            order_id = row[self.order_id_col_name].split(':')[0]
            if order_id not in orders:
                orders[order_id] = OrderInfo(
                    order_id=order_id,
                    status='',
                    order_amount=0,
                    profit_amount=0,
                    profit_amount_rub=0,
                    currency_code='USD',
                    field_values={k: v for k, v in row.items() if k not in cols_blacklist},
                )
            curr_order = orders[order_id]
            new_order_amount = curr_order.order_amount + self.price_fixer(row[self.order_amount_col_name])
            new_profit_amount = curr_order.profit_amount + self.price_fixer(row[self.profit_amount_col_name])
            new_field_values = curr_order.field_values
            new_field_values.update({self.order_amount_col_name: new_order_amount, self.profit_amount_col_name: new_profit_amount})
            orders[order_id] = OrderInfo(
                order_id=curr_order.order_id,
                status=curr_order.status,
                order_amount=new_order_amount,
                profit_amount=new_profit_amount,
                profit_amount_rub=curr_order.profit_amount_rub,
                currency_code=curr_order.currency_code,
                field_values=new_field_values,
            )
        return orders
