# -*- coding: utf-8 -*-

import logging

from copy import copy, deepcopy
import os

from openpyxl.utils import get_column_letter
import openpyxl


LOG = logging.getLogger(__name__)


class ReconciliationReport(object):

    def __init__(self, partner, debug_mode):
        self.partner = partner
        self.debug_mode = debug_mode
        self.cpa_fields = [item[0] for item in partner.report_fields]
        self.report_file_name = None

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.debug_mode:
            return
        if self.report_file_name is not None and os.path.exists(self.report_file_name):
            os.remove(self.report_file_name)

    def make_report(self, report_diff, partner_orders, cpa_orders):
        wb = openpyxl.Workbook()
        ws = wb.active
        self.write_differed_orders(ws, report_diff, cpa_orders, partner_orders)

        ws = wb.create_sheet(title='Нет в отчёте партнёра')
        self.write_cpa_extra_orders(ws, report_diff, cpa_orders)

        ws = wb.create_sheet(title='Нет в CPA')
        self.write_partner_extra_orders(ws, report_diff, partner_orders)

        fn = '{}_reconciliation_report.xlsx'.format(self.partner.name)
        wb.save(fn)
        self.report_file_name = fn
        if self.debug_mode:
            LOG.info("Reconciliation report is stored into %s", fn)

    def write_differed_orders(self, ws, report_diff, cpa_orders, partner_orders):
        ws.title = 'Отличающиеся заказы'
        col_letters = dict()
        report_fields = deepcopy(self.partner.report_fields)
        report_fields[0] = (report_fields[0][0], None)
        report_fields.insert(0, ('fail_reason', None))
        col_num = 1
        for cpa_col_name, partner_col_name in report_fields:
            first_cell_letter = get_column_letter(col_num)
            second_cell_letter = get_column_letter(col_num + 1)
            first_cell_1 = '{}1'.format(first_cell_letter)
            second_cell_1 = '{}1'.format(second_cell_letter)
            first_cell_2 = '{}2'.format(first_cell_letter)
            second_cell_2 = '{}2'.format(second_cell_letter)
            if cpa_col_name is None:
                ws[first_cell_1] = partner_col_name
                ws[first_cell_2] = 'report'
                col_letters[first_cell_letter] = partner_col_name
                col_num += 1
                continue
            elif partner_col_name is None:
                ws[first_cell_1] = cpa_col_name
                ws[first_cell_2] = 'API'
                col_letters[first_cell_letter] = cpa_col_name
                col_num += 1
                continue
            ws[first_cell_1] = partner_col_name
            ws[first_cell_2] = 'API'
            ws[second_cell_2] = 'report'
            ws.merge_cells('{}:{}'.format(first_cell_1, second_cell_1))
            col_letters[first_cell_letter] = cpa_col_name
            col_letters[second_cell_letter] = partner_col_name
            col_num += 2
        for row_num, row in enumerate(report_diff.differed_orders, start=3):
            order = copy(partner_orders[row.order_id].field_values)
            order.update(cpa_orders[row.order_id].field_values)
            order['fail_reason'] = row.fail_reason
            for letter, field in col_letters.items():
                cell = '{}{}'.format(letter, row_num)
                field = order[field]
                ws[cell].value = field

    def write_cpa_extra_orders(self, ws, report_diff, cpa_orders):
        col_names = list()
        col_num = 1
        for col_name, _ in self.partner.report_fields:
            if col_name is None:
                continue
            cell = '{}1'.format(get_column_letter(col_num))
            ws[cell] = col_name
            col_names.append(col_name)
            col_num += 1
        for row_num, order_id in enumerate(report_diff.cpa_extra_orders, start=2):
            order = cpa_orders[order_id].field_values
            for col_num, field_name in enumerate(col_names, start=1):
                letter = get_column_letter(col_num)
                cell = '{}{}'.format(letter, row_num)
                field = order[field_name]
                ws[cell].value = field

    def write_partner_extra_orders(self, ws, report_diff, partner_orders):
        col_names = list()
        for _, col_name in self.partner.report_fields:
            if col_name is None:
                continue
            col_num = len(col_names) + 1
            cell = '{}1'.format(get_column_letter(col_num))
            ws[cell] = col_name
            col_names.append(col_name)
        for row_num, order_id in enumerate(report_diff.partner_extra_orders, start=2):
            order = partner_orders[order_id].field_values
            for col_num, field_name in enumerate(col_names, start=1):
                letter = get_column_letter(col_num)
                cell = '{}{}'.format(letter, row_num)
                field = order[field_name]
                ws[cell].value = field
