# -*- coding: utf-8 -*-

from typing import List, NamedTuple
import logging

from startrek_client import Startrek
from travel.library.python.tools import with_retries


LOG = logging.getLogger(__name__)


class IssueInfo(NamedTuple):
    queue: str
    summary: str
    description: str
    assignee: str
    status: str
    attachments: List[str]
    related_issue: str


class StClient(object):

    def __init__(self, token, debug_mode):
        self.client = Startrek(useragent='python', base_url='https://st-api.yandex-team.ru', token=token)
        self.debug_mode = debug_mode

    def create_issue(self, issue_info):
        if self.debug_mode:
            LOG.debug('queue: %s', issue_info.queue)
            LOG.debug('summary: %s', issue_info.summary)
            LOG.debug('description: \n%s', issue_info.description)
            LOG.debug('assignee: %s', issue_info.assignee)
            LOG.debug('status: %s', issue_info.status)
            LOG.debug('attachments: %r', issue_info.attachments)
            LOG.debug('related_issue: %s', issue_info.related_issue)
            return
        issue = self.try_create_issue(issue_info)
        LOG.info('created task %r', issue)
        issue.links.create(issue=issue_info.related_issue, relationship='relates')
        if issue_info.status != 'open':
            with_retries(lambda: issue.transitions['close'].execute(resolution=issue_info.status))

    @with_retries
    def try_create_issue(self, issue_info):
        issue = self.client.issues.create(
            queue=issue_info.queue,
            summary=issue_info.summary,
            type={'name': 'Task'},
            description=issue_info.description,
            assignee=issue_info.assignee,
            attachments=issue_info.attachments,
        )
        LOG.info('created task %r', issue)
        return issue
