# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from argparse import ArgumentParser
from datetime import datetime
import logging.config

from yt.wrapper import YtClient
import yt.wrapper as yt

from travel.cpa.lib.common import hide_secrets
from travel.cpa.lib.lib_datetime import timestamp
from travel.cpa.lib.lib_logging import LOG_CONFIG, get_logger
from travel.cpa.lib.metered_task_executor import MeteredTaskExecutor
from travel.library.python.tools import replace_args_from_env


LOG = get_logger('calc_orders_lag')

CATEGORIES = ['avia', 'hotels', 'train', 'suburban', 'buses']


class Executor(MeteredTaskExecutor):

    def __init__(self, options):
        common_labels = {'yt_proxy': options.yt_proxy}
        super(Executor, self).__init__('calc_orders_lag', options, common_labels)
        self.yt_client = YtClient(options.yt_proxy, options.yt_token)

    def try_execute(self):
        started_at = timestamp(datetime.utcnow())

        for category in self.options.categories:
            orders_table_name = self.options.orders_table_name
            attribute_path = yt.ypath_join(self.options.yt_root, category, orders_table_name, '@_updated_at_max')
            ts = self.yt_client.get(attribute_path)
            lag = (started_at - ts) / 60
            LOG.debug('Lag for %s is %d', category, lag)
            self.solomon_client.send('orders.lag.minutes', lag, category=category)


def main():
    logging.config.dictConfig(LOG_CONFIG)

    parser = ArgumentParser()
    parser.add_argument('--yt-root', required=True)
    parser.add_argument('--yt-proxy', required=True)
    parser.add_argument('--yt-token', required=True)

    parser.add_argument('--categories', nargs='*', default=CATEGORIES)
    parser.add_argument('--orders-table-name', default='orders')

    Executor.configure(parser)
    options = parser.parse_args(args=replace_args_from_env())

    displayed_options = hide_secrets(vars(options))
    LOG.info('Working with %r', displayed_options)

    Executor.resolve_secrets(options)
    Executor(options).execute()


if __name__ == '__main__':
    main()
