package ru.yandex.travel.cpa.data_processing.flow.logbroker;

import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import com.google.common.util.concurrent.MoreExecutors;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;

@Slf4j
public class MultiHostLogbrokerReader implements LogbrokerReader {
    private final LogbrokerConnectionProperties connectionProperties;
    private final LogbrokerProperties logbrokerProperties;
    private final ExecutorService executorService;
    private final AtomicBoolean closed = new AtomicBoolean();
    private final LinkedBlockingQueue<LogbrokerDataBatch> batchQueue;
    private final boolean writeMessagesToLog;

    public MultiHostLogbrokerReader(
            LogbrokerConnectionProperties connectionProperties,
            LogbrokerProperties logbrokerProperties,
            Map<String, String> mdcContext,
            boolean writeMessagesToLog
    ) {
        this.connectionProperties = connectionProperties;
        this.logbrokerProperties = logbrokerProperties;
        this.writeMessagesToLog = writeMessagesToLog;
        int readersCount = connectionProperties.getHosts().size();
        batchQueue = new LinkedBlockingQueue<>(readersCount);
        executorService = Executors.newFixedThreadPool(readersCount);
        startThreads(mdcContext);
    }

    public LogbrokerDataBatch getBatch() throws Exception {
        var batch = batchQueue.poll();
        while (!closed.get() && batch == null) {
            Thread.sleep(connectionProperties.getQueueAccessTimeout().toMillis());
            batch = batchQueue.poll();
        }
        return batch;
    }

    @Override
    public void close() {
        closed.set(true);
        MoreExecutors.shutdownAndAwaitTermination(executorService, 10, TimeUnit.SECONDS);
    }

    private void startThreads(Map<String, String> mdcContext) {
        for (var host : connectionProperties.getHosts().entrySet()) {
            executorService.submit(() -> {
                MDC.setContextMap(mdcContext);
                var cluster = host.getKey();
                MDC.put("logbroker.cluster", cluster);
                try {
                    readCluster(host.getValue(), cluster);
                } catch (InterruptedException e) {
                    log.info("Reader {} thread was interrupted", host);
                    Thread.currentThread().interrupt();
                }
                MDC.clear();
            });
        }

    }

    private void readCluster(String host, String cluster) throws InterruptedException {
        var timeoutMillis = connectionProperties.getQueueAccessTimeout().toMillis();
        while (!closed.get()) {
            try (var reader = new SingleHostLogbrokerReader(
                    connectionProperties,
                    logbrokerProperties,
                    host,
                    cluster,
                    writeMessagesToLog
            )) {
                while (!closed.get()) {
                    var batch = reader.getBatch();
                    if (batch.isEmpty()) {
                        continue;
                    }
                    while (!closed.get()) {
                        if (batchQueue.offer(batch, timeoutMillis, TimeUnit.MILLISECONDS)) {
                            break;
                        }
                    }
                }
            } catch (IllegalStateException e) {
                log.warn("Execution exception", e);
                waitWithInterrupt(connectionProperties.getReaderRestartTimeout().toMillis());
            }
        }
    }

    private void waitWithInterrupt(long periodMillis) throws InterruptedException {
        long waitStart = System.currentTimeMillis();
        while (!closed.get()) {
            if (System.currentTimeMillis() - waitStart >= periodMillis) {
                break;
            }
            Thread.sleep(1000);
        }
    }
}
