package ru.yandex.travel.cpa.data_processing.flow.logbroker;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.kikimr.persqueue.LogbrokerClientFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.kikimr.persqueue.consumer.sync.SyncConsumerConfig;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerInitResponse;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerReadResponse;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageBatch;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageData;
import ru.yandex.kikimr.persqueue.proxy.ProxyBalancer;

@Slf4j
public class SingleHostLogbrokerReader implements LogbrokerReader {
    private final LogbrokerConnectionProperties connectionProperties;
    private final ProxyBalancer proxyBalancer;
    private final LogbrokerClientFactory clientFactory;
    private final SyncConsumerConfig syncConsumerConfig;
    private final String cluster;
    private SyncConsumer syncConsumer;
    private final boolean writeMessagesToLog;

    public SingleHostLogbrokerReader(
            LogbrokerConnectionProperties connectionProperties,
            LogbrokerProperties logbrokerProperties,
            String host,
            String cluster,
            boolean writeMessagesToLog
    ) {
        this.connectionProperties = connectionProperties;
        this.cluster = cluster;
        proxyBalancer = new ProxyBalancer(host, connectionProperties.getPort(), connectionProperties.getPort());
        this.clientFactory = new LogbrokerClientFactory(proxyBalancer);
        this.syncConsumerConfig = buildSyncConsumerConfig(connectionProperties, logbrokerProperties);
        this.writeMessagesToLog = writeMessagesToLog;
    }

    @Override
    public LogbrokerDataBatch getBatch() throws IllegalStateException, InterruptedException {
        if (syncConsumer == null || syncConsumer.isClosed()) {
            syncConsumer = tryGetSyncConsumer();
        }
        if (syncConsumer == null) {
            throw new IllegalStateException("Failed to get consumer");
        }

        List<LogbrokerMessage> batchData = new ArrayList<>();
        List<Long> cookies = new ArrayList<>();

        var readStart = System.currentTimeMillis();

        while (true) {
            if (batchData.size() >= connectionProperties.getMessagesPerRequest()) {
                break;
            }

            if (System.currentTimeMillis() - readStart >= connectionProperties.getTimeout().toMillis()) {
                break;
            }

            if (cookies.size() >= connectionProperties.getMaxReadsPerBatch()) {
                break;
            }

            ConsumerReadResponse read = syncConsumer.read();
            if (read == null) {
                break;
            }

            cookies.add(read.getCookie());
            for (MessageBatch batch : read.getBatches()) {
                var messageData = batch.getMessageData();
                for (MessageData data : messageData) {
                    var decompressedData = data.getDecompressedData();
                    String sourceId  = new String(data.getMessageMeta().getSourceId(), StandardCharsets.UTF_8);
                    batchData.add(LogbrokerMessage.builder()
                            .bytes(decompressedData)
                            .sourceId(sourceId)
                            .build()
                    );
                    if (writeMessagesToLog) {
                        log.debug("topic:{} incoming message::{}", batch.getTopic(), new String(decompressedData, StandardCharsets.UTF_8));
                    }
                }
            }
        }

        if (batchData.size() == 0) {
            log.debug("read nothing");
            return LogbrokerDataBatch.emptyBatch();
        }

        log.info("read messages: {}, cookies {}", batchData.size(), cookies.size());
        return new LogbrokerDataBatch(cookies, batchData, this::commit, cluster);
    }

    @Override
    public void close() throws InterruptedException {
        closeSyncConsumer();
        closeProxyBalancer();
    }

    private void commit(LogbrokerDataBatch batch) throws RuntimeException {
        var cookies = batch.getCookies();
        var cookiesCount = cookies.size();
        int retries = connectionProperties.getRetries();
        while (true) {
            try {
                syncConsumer.commit(cookies);
                log.info("committed cookies: {}", cookiesCount);
                break;
            } catch (Exception e) {
                if (retries > 0) {
                    log.warn("failed to commit {} cookie(s). {} tries left", cookiesCount, retries, e);
                    retries -= 1;
                    continue;
                }
                throw new IllegalStateException("Failed to commit cookie", e);
            }
        }
    }

    private SyncConsumer tryGetSyncConsumer() {
        int retries = connectionProperties.getRetries();
        while (true) {
            try {
                syncConsumer = clientFactory.syncConsumer(syncConsumerConfig);
                ConsumerInitResponse init = syncConsumer.init();
                log.info("Sync consumer initialized, sessionId={}", init.getSessionId());
                return syncConsumer;
            } catch (Exception e) {
                if (retries > 0) {
                    log.warn("Failed to get consumer. {} tries left", retries, e);
                    retries -= 1;
                    continue;
                }
                throw new IllegalStateException("Failed to get consumer", e);
            }
        }
    }

    private static SyncConsumerConfig buildSyncConsumerConfig(
            LogbrokerConnectionProperties connectionProperties,
            LogbrokerProperties lbProperties
    ) {
        var topics = new ArrayList<>(lbProperties.getTopic());

        var readDataOnlyAfterTimestampMs = System.currentTimeMillis() -
                connectionProperties.getReadingWindow().toMillis();

        var configBuilder = SyncConsumerConfig.builder(topics, lbProperties.getConsumer())
                .setInitTimeout((int) connectionProperties.getInitTimeout().toSeconds(), TimeUnit.SECONDS)
                .setReadDataTimeout((int) connectionProperties.getTimeout().toSeconds(), TimeUnit.SECONDS)
                .configureReader((readerConfig) -> readerConfig
                        .setMaxCount(connectionProperties.getMessagesPerRequest())
                        .setReadDataOnlyAfterTimestampMs(readDataOnlyAfterTimestampMs)
                );
        var token = connectionProperties.getToken();
        if (token == null) {
            configBuilder.setCredentialsProvider(null);
        } else {
            configBuilder.setCredentialsProvider(() -> Credentials.oauth(token));
        }
        return configBuilder.build();
    }

    private void closeSyncConsumer() {
        if (syncConsumer == null) {
            log.warn("SyncConsumer already closed");
            return;
        }
        var closeStart = System.currentTimeMillis();
        syncConsumer.close();
        while (true) {
            if (syncConsumer.isClosed()) {
                break;
            }
            if (System.currentTimeMillis() - closeStart >= connectionProperties.getConsumerCloseTimeout().toMillis()) {
                break;
            }
        }
        if (syncConsumer.isClosed()) {
            log.info("SyncConsumer closed successfully");
        }
        else {
            log.warn("SyncConsumer failed to close");
        }
        syncConsumer = null;
    }

    private void closeProxyBalancer() throws InterruptedException {
        int timeout = (int) connectionProperties.getConsumerCloseTimeout().toMillis();
        if (proxyBalancer.shutdown(timeout, TimeUnit.MILLISECONDS)) {
            log.info("Proxy balancer shutdown succeed");
        }
        else {
            log.info("Proxy balancer shutdown failed");
        }
    }
}
