package ru.yandex.travel.cpa.data_processing.flow.management;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.stereotype.Component;

import ru.yandex.travel.cpa.data_processing.flow.FlowApplicationProperties;

@Component
@EnableConfigurationProperties({FlowApplicationProperties.class})
@Slf4j
public class ShutdownManager implements HealthIndicator {
    private final FlowApplicationProperties flowApplicationProperties;
    private final ApplicationContext context;
    private final ScheduledExecutorService shutdownSequenceExecutor;
    private final Map<String, Boolean> serviceState = new HashMap<>();
    private final AtomicBoolean shutdownInitiated = new AtomicBoolean(false);
    private Long shutdownTimestamp;

    ShutdownManager(FlowApplicationProperties flowApplicationProperties, ApplicationContext context) {
        this.flowApplicationProperties = flowApplicationProperties;
        this.context = context;
        shutdownSequenceExecutor = new ScheduledThreadPoolExecutor(
                1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("shutdown-thread-%s").build()
        );
    }

    public void initiateShutdown() {
        if (shutdownInitiated.getAndSet(true)) {
            log.info("Shutdown already scheduled");
            return;
        }
        shutdownTimestamp = System.currentTimeMillis();
        shutdownSequenceExecutor.schedule(
                () -> {
                    log.info("Starting context close");
                    ((ConfigurableApplicationContext) context).close();
                    log.info("Finished context close");
                    System.exit(0);
                },
                flowApplicationProperties.getShutdownDelay().toMillis(),
                TimeUnit.MILLISECONDS
        );
        log.info("Shutdown scheduled");
    }

    @Override
    public Health health() {
        if (shutdownInitiated.get()) {
            return Health.down().withDetail("ShutdownInitiated", shutdownTimestamp).build();
        } else {
            return Health.up().build();
        }
    }

    synchronized public void registerService(String serviceName) {
        log.info("Registering service {}", serviceName);
        serviceState.put(serviceName, false);
    }

    synchronized public void notifyServiceFinished(String serviceName) {
        log.info("{} finished", serviceName);
        serviceState.put(serviceName, true);
        boolean allFinished = true;
        for (var state : serviceState.values()) {
            if (!state) {
                allFinished = false;
                break;
            }
        }
        if (allFinished) {
            log.info("All services finished. Requesting shutdown");
            initiateShutdown();
        }
    }
}
