package ru.yandex.travel.cpa.data_processing.flow.model.snapshots;

import java.io.IOException;
import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.travel.cpa.data_processing.flow.model.MessageDecoder;
import ru.yandex.travel.cpa.data_processing.flow.model.orders.OrderKey;
import ru.yandex.travel.cpa.data_processing.flow.yt.YtMessageConverter;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedValue;

public class SnapshotsConverter implements YtMessageConverter<OrderKey, Snapshot> {
    private final static int DATA_COL = 3;

    private final MessageDecoder<Snapshot> snapshotDecoder = new SnapshotJsonDecoder();

    @Override
    public TableSchema getTableSchema() {
        return new TableSchema.Builder()
                .addKey("partner_name", ColumnValueType.STRING)
                .addKey("partner_order_id", ColumnValueType.STRING)
                .addKey("updated_at", ColumnValueType.UINT64)
                .addValue("data", ColumnValueType.STRING)
                .build();
    }

    @Override
    public Map<String, Serializable> serialize(Snapshot object) {
        var serialized = new HashMap<String, Serializable>();
        serialized.put("partner_name", object.getPartnerName());
        serialized.put("partner_order_id", object.getPartnerOrderId());
        serialized.put("updated_at", object.getUpdatedAt());
        serialized.put("data", object.getData());
        return serialized;
    }

    // column order should match one at getLookupColumns
    @Override
    public Snapshot deserialize(List<UnversionedValue> row) throws IOException {
        var data = row.get(DATA_COL).bytesValue();
        var fields = snapshotDecoder.decode(data).get(0);
        return new Snapshot(
                fields.getPartnerName(),
                fields.getPartnerOrderId(),
                fields.getHash(),
                fields.getUpdatedAt(),
                data
        );
    }

    // column order should match one at deserialize
    @Override
    public List<String> getLookupColumns() {
        return List.of("partner_name", "partner_order_id", "updated_at", "data");
    }

    @Override
    public List<Object> getFilterValues(Snapshot message) {
        return List.of(message.getPartnerName(), message.getPartnerOrderId(), message.getUpdatedAt());
    }

    @Override
    public List<String> getSelectColumns() {
        return List.of("partner_name", "partner_order_id");
    }

    @Override
    public String getSelectValues(OrderKey key) {
        return String.format("('%s', '%s')", key.getPartnerName(), key.getPartnerOrderId());
    }
}
