package ru.yandex.travel.cpa.data_processing.flow.processors;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Timer;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.cpa.data_processing.flow.logbroker.LogbrokerDataBatch;
import ru.yandex.travel.cpa.data_processing.flow.model.MessageDecoder;
import ru.yandex.travel.cpa.data_processing.flow.model.labels.Label;
import ru.yandex.travel.cpa.data_processing.flow.model.labels.LabelKey;
import ru.yandex.travel.cpa.data_processing.flow.model.orders.OrderKey;
import ru.yandex.travel.cpa.data_processing.flow.model.orders.OrderPurgatoryKey;
import ru.yandex.travel.cpa.data_processing.flow.model.orders.OrderPurgatoryValue;
import ru.yandex.travel.cpa.data_processing.flow.yt.SyncYtClient;

@Slf4j
public class LabelProcessor implements Processor {
    private final static String BREAKER_LABEL = "PLEASE_STOP";
    private boolean processingFinished = false;
    private final MessageDecoder<Label> labelDecoder;
    private final SyncYtClient<LabelKey, Label> labelClient;
    private final SyncYtClient<OrderKey, OrderKey> orderQueueClient;
    private final SyncYtClient<OrderPurgatoryKey, OrderPurgatoryValue> orderPurgatoryClient;
    private final AtomicBoolean isClosed = new AtomicBoolean(false);

    private final Counter incomingNewLabelCounter;
    private final Counter incomingOldLabelCounter;
    private final Timer readLagDistribution;

    public LabelProcessor(
            MessageDecoder<Label> labelDecoder,
            SyncYtClient<LabelKey, Label> labelClient,
            SyncYtClient<OrderKey, OrderKey> orderQueueClient,
            SyncYtClient<OrderPurgatoryKey, OrderPurgatoryValue> orderPurgatoryClient
    ) {
        this.labelDecoder = labelDecoder;
        this.labelClient = labelClient;
        this.orderQueueClient = orderQueueClient;
        this.orderPurgatoryClient = orderPurgatoryClient;

        incomingNewLabelCounter = Counter
                .builder("cpa.flow.incomingLabelCount")
                .tag("status", "new")
                .register(Metrics.globalRegistry);

        incomingOldLabelCounter = Counter
                .builder("cpa.flow.incomingLabelCount")
                .tag("status", "old")
                .register(Metrics.globalRegistry);

        readLagDistribution = Timer.builder("cpa.flow.labelReadLagDistribution")
                .serviceLevelObjectives(
                        Duration.ofSeconds(1),
                        Duration.ofSeconds(5),
                        Duration.ofSeconds(30),
                        Duration.ofMinutes(1),
                        Duration.ofMinutes(5),
                        Duration.ofMinutes(10),
                        Duration.ofMinutes(30),
                        Duration.ofHours(1),
                        Duration.ofHours(2),
                        Duration.ofHours(3),
                        Duration.ofHours(4),
                        Duration.ofHours(5),
                        Duration.ofDays(1),
                        Duration.ofDays(2)
                )
                .register(Metrics.globalRegistry);
    }

    public boolean process(LogbrokerDataBatch snapshots) throws Exception {
        while (!isClosed.get()) {
            try {
                tryProcess(snapshots);
                return true;
            } catch (ExecutionException | TimeoutException e) {
                log.warn("YT interaction error", e);
            }
        }
        return false;
    }

    public boolean isProcessingFinished() {
        return processingFinished;
    }

    public void close() {
        isClosed.set(true);
    }

    private void tryProcess(LogbrokerDataBatch labels) throws Exception {
        var convertedLabels = getConvertedLabels(labels);
        var newLabels = getNewLabels(convertedLabels);
        var ordersToSend = getOrdersToSend(getPurgatoryOrders(newLabels));
        try (var transaction = labelClient.getTransaction()) {
            labelClient.send(convertedLabels, transaction);
            if (!ordersToSend.isEmpty()) {
                orderQueueClient.send(ordersToSend);
            }
            labelClient.commitTransaction(transaction);
        }

        var incomingLabels = convertedLabels
                .stream()
                .map(l -> String.format("\"%s\"", l.getLabel()))
                .collect(Collectors.toList());
        log.debug("incoming labels::{}", incomingLabels);

        var newLabelCount = newLabels.size();
        var processedLabelCount = convertedLabels.size() - newLabels.size();
        incomingNewLabelCounter.increment(newLabelCount);
        incomingOldLabelCounter.increment(processedLabelCount);
        log.info("labels new: {}, old: {}", newLabelCount, processedLabelCount);
    }

    private List<Label> getConvertedLabels(LogbrokerDataBatch labels) throws java.io.IOException {
        long batchProcessingStartTime = System.currentTimeMillis() / 1000;
        var convertedLabels = new ArrayList<Label>();
        long maxLag = 0;
        for (var message : labels.getMessages()) {
            for (var convertedLabel : labelDecoder.decode(message.getBytes())) {
                if (convertedLabel.getLabel() == null) {
                    log.warn("Got label with null key");
                    continue;
                }
                var timestamp = convertedLabel.getTimestamp();
                var labelLag = batchProcessingStartTime - timestamp;
                readLagDistribution.record(Duration.ofSeconds(labelLag));

                if (labelLag > maxLag) {
                    maxLag = labelLag;
                }

                if (convertedLabel.getLabel().equals(BREAKER_LABEL)) {
                    log.info("Got breaker label");
                    processingFinished = true;
                    break;
                }
                convertedLabels.add(convertedLabel);
            }
        }
        log.debug("batch lag max: {}", Duration.ofSeconds(maxLag));
        return convertedLabels;
    }

    private List<Label> getNewLabels(List<Label> labels) throws Exception {
        var existingLabels = new HashMap<String, Label>();
        for (var label : labelClient.lookup(labels)) {
            existingLabels.put(label.getLabel(), label);
        }
        var newLabels = new ArrayList<Label>();
        for (var label : labels) {
            var existingLabel = existingLabels.get(label.getLabel());
            if (existingLabel != null && label.getCategory().equals(existingLabel.getCategory())) {
                continue;
            }
            newLabels.add(label);
        }
        return newLabels;
    }

    private List<OrderPurgatoryValue> getPurgatoryOrders(List<Label> labels) throws Exception {
        var orderKeys = new ArrayList<OrderPurgatoryKey>();
        for (var label : labels) {
            orderKeys.add(new OrderPurgatoryKey(label.getLabel(), null, null));
        }
        return orderPurgatoryClient.select(orderKeys);
    }

    private List<OrderKey> getOrdersToSend(List<OrderPurgatoryValue> orders) {
        var ordersToSend = new ArrayList<OrderKey>();
        for (var order : orders) {
            ordersToSend.add(new OrderKey(order.getPartner_name(), order.getPartner_order_id()));
        }
        return ordersToSend;
    }
}
