package ru.yandex.travel.cpa.data_processing.flow.services;

import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;

import io.micrometer.core.instrument.Counter;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import ru.yandex.travel.cpa.data_processing.flow.logbroker.LogbrokerConnectionProperties;
import ru.yandex.travel.cpa.data_processing.flow.logbroker.LogbrokerProperties;
import ru.yandex.travel.cpa.data_processing.flow.logbroker.MultiHostLogbrokerReader;
import ru.yandex.travel.cpa.data_processing.flow.management.ShutdownManager;
import ru.yandex.travel.cpa.data_processing.flow.processors.Processor;

@Slf4j
public abstract class AbstractService implements InitializingBean, DisposableBean {

    final AtomicBoolean active = new AtomicBoolean();
    ShutdownManager shutdownManager;

    void run(
            String processorName,
            LogbrokerConnectionProperties logbrokerConnectionProperties,
            LogbrokerProperties logbrokerProperties,
            Processor processor,
            Counter handledErrorsCounter,
            Counter unhandledErrorsCounter,
            Map<String, String> mdcContext,
            boolean writeMessagesToLog
    ) {
        MDC.setContextMap(mdcContext);
        MDC.put("processor", processorName);
        String serviceName = String.format("%s:%s", this.getClass().getName(), processorName);
        shutdownManager.registerService(serviceName);
        log.info("processing started");
        active.set(true);
        while (active.get()) {
            try (var reader = new MultiHostLogbrokerReader(
                    logbrokerConnectionProperties,
                    logbrokerProperties,
                    MDC.getCopyOfContextMap(),
                    writeMessagesToLog
            )) {
                while (active.get()) {
                    var batch = reader.getBatch();
                    if (batch.isEmpty()) {
                        continue;
                    }
                    MDC.put("logbroker.cluster", batch.getCluster());
                    if (processor.process(batch)) {
                        batch.commit();
                    }
                    if (processor.isProcessingFinished()) {
                        active.set(false);
                    }
                }
            } catch (ExecutionException | TimeoutException | IllegalStateException e) {
                log.warn("Execution exception", e);
                handledErrorsCounter.increment();
            } catch (Exception e) {
                log.error("Unhandled error while processing batch", e);
                unhandledErrorsCounter.increment();
                break;
            }
        }
        shutdownManager.notifyServiceFinished(serviceName);
        MDC.clear();
    }
}
