package ru.yandex.travel.cpa.data_processing.flow.services;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import com.google.common.util.concurrent.MoreExecutors;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.cpa.data_processing.flow.FlowApplicationProperties;
import ru.yandex.travel.cpa.data_processing.flow.management.ShutdownManager;
import ru.yandex.travel.cpa.data_processing.flow.model.labels.Label;
import ru.yandex.travel.cpa.data_processing.flow.model.labels.LabelConverter;
import ru.yandex.travel.cpa.data_processing.flow.model.labels.LabelJsonDecoderAvia;
import ru.yandex.travel.cpa.data_processing.flow.model.labels.LabelJsonDecoderBuses;
import ru.yandex.travel.cpa.data_processing.flow.model.labels.LabelJsonDecoderHotels;
import ru.yandex.travel.cpa.data_processing.flow.model.labels.LabelJsonDecoderSuburban;
import ru.yandex.travel.cpa.data_processing.flow.model.labels.LabelJsonDecoderTours;
import ru.yandex.travel.cpa.data_processing.flow.model.labels.LabelJsonDecoderTrain;
import ru.yandex.travel.cpa.data_processing.flow.model.labels.LabelKey;
import ru.yandex.travel.cpa.data_processing.flow.model.orders.OrderKey;
import ru.yandex.travel.cpa.data_processing.flow.model.orders.OrderPurgatoryConverter;
import ru.yandex.travel.cpa.data_processing.flow.model.orders.OrderPurgatoryKey;
import ru.yandex.travel.cpa.data_processing.flow.model.orders.OrderPurgatoryValue;
import ru.yandex.travel.cpa.data_processing.flow.model.orders.OrderQueueConverter;
import ru.yandex.travel.cpa.data_processing.flow.processors.LabelProcessor;
import ru.yandex.travel.cpa.data_processing.flow.yt.SyncYtClient;
import ru.yandex.travel.cpa.data_processing.flow.yt.SyncYtClientFactory;

@Component
@EnableConfigurationProperties({
        FlowApplicationProperties.class,
        LabelServiceProperties.class
})
@Slf4j
public class LabelService extends AbstractService {

    private final FlowApplicationProperties flowApplicationProperties;
    private final LabelServiceProperties labelServiceProperties;
    private ExecutorService executorService;
    private final SyncYtClient<LabelKey, Label> labelClient;
    private final SyncYtClient<OrderKey, OrderKey> orderQueueClient;
    private final SyncYtClient<OrderPurgatoryKey, OrderPurgatoryValue> orderPurgatoryClient;
    private final LabelProcessor aviaProcessor;
    private final LabelProcessor hotelsProcessor;
    private final LabelProcessor trainProcessor;
    private final LabelProcessor suburbanProcessor;
    private final LabelProcessor toursProcessor;
    private final LabelProcessor busesProcessor;

    private final Counter handledErrorsCounter;
    private final Counter unhandledErrorsCounter;

    public LabelService(
            FlowApplicationProperties flowApplicationProperties,
            LabelServiceProperties labelServiceProperties,
            ShutdownManager shutdownManager
    ) {
        this.flowApplicationProperties = flowApplicationProperties;
        this.labelServiceProperties = labelServiceProperties;
        this.shutdownManager = shutdownManager;

        handledErrorsCounter = Counter
                .builder("cpa.flow.errorsCount")
                .tag("type", "handled")
                .register(Metrics.globalRegistry);
        unhandledErrorsCounter = Counter
                .builder("cpa.flow.errorsCount")
                .tag("type", "unhandled")
                .register(Metrics.globalRegistry);
        var ytErrorsCounter = Counter
                .builder("cpa.flow.errorsCount")
                .tag("type", "ytInteraction")
                .register(Metrics.globalRegistry);

        var syncYtClientFactory = new SyncYtClientFactory(labelServiceProperties.getYtConnection(), ytErrorsCounter);

        var ytTablesProperties = labelServiceProperties.getYtTables();
        labelClient = syncYtClientFactory.newSyncYtClient(
                ytTablesProperties.getLabels(),
                new LabelConverter()
        );
        orderQueueClient = syncYtClientFactory.newSyncYtClient(
                ytTablesProperties.getOrderQueue(),
                new OrderQueueConverter()
        );
        orderPurgatoryClient = syncYtClientFactory.newSyncYtClient(
                ytTablesProperties.getOrderPurgatory(),
                new OrderPurgatoryConverter()
        );

        aviaProcessor = new LabelProcessor(
                new LabelJsonDecoderAvia(),
                labelClient,
                orderQueueClient,
                orderPurgatoryClient
        );

        hotelsProcessor = new LabelProcessor(
                new LabelJsonDecoderHotels(),
                labelClient,
                orderQueueClient,
                orderPurgatoryClient
        );

        trainProcessor = new LabelProcessor(
                new LabelJsonDecoderTrain(),
                labelClient,
                orderQueueClient,
                orderPurgatoryClient
        );

        suburbanProcessor = new LabelProcessor(
                new LabelJsonDecoderSuburban(),
                labelClient,
                orderQueueClient,
                orderPurgatoryClient
        );

        toursProcessor = new LabelProcessor(
                new LabelJsonDecoderTours(),
                labelClient,
                orderQueueClient,
                orderPurgatoryClient
        );

        busesProcessor = new LabelProcessor(
                new LabelJsonDecoderBuses(),
                labelClient,
                orderQueueClient,
                orderPurgatoryClient
        );
    }

    public void afterPropertiesSet() {
        MDC.put("serviceName", "label");
        var contextMap = MDC.getCopyOfContextMap();

        executorService = Executors.newCachedThreadPool();
        executorService.submit(() -> run(
                "avia",
                labelServiceProperties.getLbConnection(),
                labelServiceProperties.getLogbrokerAvia(),
                aviaProcessor,
                handledErrorsCounter,
                unhandledErrorsCounter,
                contextMap,
                true
        ));
        executorService.submit(() -> run(
                "hotels",
                labelServiceProperties.getLbConnection(),
                labelServiceProperties.getLogbrokerHotels(),
                hotelsProcessor,
                handledErrorsCounter,
                unhandledErrorsCounter,
                contextMap,
                true
        ));
        executorService.submit(() -> run(
                "train",
                labelServiceProperties.getLbConnection(),
                labelServiceProperties.getLogbrokerTrain(),
                trainProcessor,
                handledErrorsCounter,
                unhandledErrorsCounter,
                contextMap,
                true
        ));
        executorService.submit(() -> run(
                "suburban",
                labelServiceProperties.getLbConnection(),
                labelServiceProperties.getLogbrokerSuburban(),
                suburbanProcessor,
                handledErrorsCounter,
                unhandledErrorsCounter,
                contextMap,
                true
        ));
        executorService.submit(() -> run(
                "tours",
                labelServiceProperties.getLbConnection(),
                labelServiceProperties.getLogbrokerTours(),
                toursProcessor,
                handledErrorsCounter,
                unhandledErrorsCounter,
                contextMap,
                true
        ));
        executorService.submit(() -> run(
                "buses",
                labelServiceProperties.getLbConnection(),
                labelServiceProperties.getLogbrokerBuses(),
                busesProcessor,
                handledErrorsCounter,
                unhandledErrorsCounter,
                contextMap,
                true
        ));
    }

    public void destroy() {
        log.info("Closing service");
        active.set(false);
        aviaProcessor.close();
        hotelsProcessor.close();
        trainProcessor.close();
        suburbanProcessor.close();
        toursProcessor.close();
        busesProcessor.close();
        MoreExecutors.shutdownAndAwaitTermination(
                executorService,
                flowApplicationProperties.getShutdownTimeout().toMillis(),
                TimeUnit.MILLISECONDS
        );
        labelClient.close();
        orderQueueClient.close();
        orderPurgatoryClient.close();
        log.info("Service closed");
        MDC.clear();
    }
}
