package ru.yandex.travel.cpa.data_processing.flow.services;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import com.google.common.util.concurrent.MoreExecutors;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.cpa.data_processing.flow.FlowApplicationProperties;
import ru.yandex.travel.cpa.data_processing.flow.management.ShutdownManager;
import ru.yandex.travel.cpa.data_processing.flow.model.orders.OrderKey;
import ru.yandex.travel.cpa.data_processing.flow.model.orders.OrderQueueConverter;
import ru.yandex.travel.cpa.data_processing.flow.model.snapshots.ProcessedSnapshotsConverter;
import ru.yandex.travel.cpa.data_processing.flow.model.snapshots.Snapshot;
import ru.yandex.travel.cpa.data_processing.flow.model.snapshots.SnapshotError;
import ru.yandex.travel.cpa.data_processing.flow.model.snapshots.SnapshotErrorsConverter;
import ru.yandex.travel.cpa.data_processing.flow.model.snapshots.SnapshotsConverter;
import ru.yandex.travel.cpa.data_processing.flow.processors.SnapshotProcessor;
import ru.yandex.travel.cpa.data_processing.flow.yt.SyncYtClient;
import ru.yandex.travel.cpa.data_processing.flow.yt.SyncYtClientFactory;

@Component
@EnableConfigurationProperties({
        FlowApplicationProperties.class,
        SnapshotServiceProperties.class
})
@Slf4j
public class SnapshotService extends AbstractService {

    private final FlowApplicationProperties flowApplicationProperties;
    private final SnapshotServiceProperties snapshotServiceProperties;
    private ExecutorService executorService;
    private final SyncYtClient<OrderKey, Snapshot> processedSnapshotsClient;
    private final SyncYtClient<OrderKey, Snapshot> snapshotsClient;
    private final SyncYtClient<OrderKey, SnapshotError> snapshotErrorsClient;
    private final SyncYtClient<OrderKey, OrderKey> orderQueueClient;
    private final SyncYtClient<OrderKey, OrderKey> slowOrderQueueClient;
    private final SnapshotProcessor snapshotProcessor;

    private final Counter handledErrorsCounter;
    private final Counter unhandledErrorsCounter;

    public SnapshotService(
            FlowApplicationProperties flowApplicationProperties,
            SnapshotServiceProperties snapshotServiceProperties,
            ShutdownManager shutdownManager
    ) {
        this.flowApplicationProperties = flowApplicationProperties;
        this.snapshotServiceProperties = snapshotServiceProperties;
        this.shutdownManager = shutdownManager;

        handledErrorsCounter = Counter
                .builder("cpa.flow.errorsCount")
                .tag("type", "handled")
                .register(Metrics.globalRegistry);
        unhandledErrorsCounter = Counter
                .builder("cpa.flow.errorsCount")
                .tag("type", "unhandled")
                .register(Metrics.globalRegistry);
        var ytErrorsCounter = Counter
                .builder("cpa.flow.errorsCount")
                .tag("type", "ytInteraction")
                .register(Metrics.globalRegistry);

        var syncYtClientFactory = new SyncYtClientFactory(
                snapshotServiceProperties.getYtConnection(), ytErrorsCounter
        );

        var ytTablesProperties = snapshotServiceProperties.getYtTables();
        processedSnapshotsClient = syncYtClientFactory.newSyncYtClient(
                ytTablesProperties.getProcessedSnapshots(),
                new ProcessedSnapshotsConverter()
        );
        snapshotsClient = syncYtClientFactory.newSyncYtClient(
                ytTablesProperties.getSnapshots(),
                new SnapshotsConverter()
        );
        snapshotErrorsClient = syncYtClientFactory.newSyncYtClient(
                ytTablesProperties.getSnapshotErrors(),
                new SnapshotErrorsConverter()
        );
        orderQueueClient = syncYtClientFactory.newSyncYtClient(
                ytTablesProperties.getOrderQueue(),
                new OrderQueueConverter()
                );
        slowOrderQueueClient = syncYtClientFactory.newSyncYtClient(
                ytTablesProperties.getSlowOrderQueue(),
                new OrderQueueConverter()
        );

        snapshotProcessor = new SnapshotProcessor(
                processedSnapshotsClient,
                snapshotsClient,
                snapshotErrorsClient,
                orderQueueClient,
                slowOrderQueueClient
        );
    }

    public void afterPropertiesSet() {
        MDC.put("serviceName", "snapshot");
        var contextMap = MDC.getCopyOfContextMap();

        executorService = Executors.newSingleThreadExecutor();
        executorService.submit(() -> run(
                "",
                snapshotServiceProperties.getLbConnection(),
                snapshotServiceProperties.getLogbroker(),
                snapshotProcessor,
                handledErrorsCounter,
                unhandledErrorsCounter,
                contextMap,
                false
        ));
    }

    public void destroy() {
        log.info("Closing service");
        active.set(false);
        snapshotProcessor.close();
        MoreExecutors.shutdownAndAwaitTermination(
                executorService,
                flowApplicationProperties.getShutdownTimeout().toMillis(),
                TimeUnit.MILLISECONDS
        );
        processedSnapshotsClient.close();
        snapshotsClient.close();
        snapshotErrorsClient.close();
        orderQueueClient.close();
        log.info("Service closed");
        MDC.clear();
    }
}
