package ru.yandex.travel.cpa.data_processing.flow.yt;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.springframework.retry.support.RetryTemplate;

import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.LookupRowsRequest;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.proxy.YtClient;

public class SyncYtClient<K, V> implements Closeable {

    private final YtClient client;
    private final YtTableProperties tableProperties;
    private final ApiServiceTransactionOptions transactionOptions;
    private final YtMessageConverter<K, V> messageConverter;
    private final RetryTemplate retryTemplate;
    private final long requestTimeout;

    public SyncYtClient(
            YtClient client,
            YtConnectionProperties connectionProperties,
            YtTableProperties tableProperties,
            YtMessageConverter<K, V> messageConverter,
            RetryTemplate retryTemplate
    ) {
        this.client = client;
        this.tableProperties = tableProperties;
        transactionOptions = new ApiServiceTransactionOptions(ETransactionType.TT_TABLET).setSticky(true);
        this.messageConverter = messageConverter;
        this.retryTemplate = retryTemplate;
        requestTimeout = connectionProperties.getRequestTimeout().toSeconds();
    }

    public ApiServiceTransaction getTransaction() throws Exception {
        return retryTemplate.execute(
                c -> client.startTransaction(transactionOptions).get(requestTimeout, TimeUnit.SECONDS)
        );
    }

    public void commitTransaction(ApiServiceTransaction transaction) throws Exception {
        retryTemplate.execute(c -> transaction.commit().get(requestTimeout, TimeUnit.SECONDS));
    }

    public void send(List<V> messages) throws Exception {
        if (messages.isEmpty()) {
            return;
        }
        try (var transaction = getTransaction()) {
            send(messages, transaction);
            commitTransaction(transaction);
        }
    }

    public void send(List<V> messages, ApiServiceTransaction transaction) throws Exception {
        if (messages.isEmpty()) {
            return;
        }
        var request = getInsertRequest(messages);
        retryTemplate.execute(c -> transaction.modifyRows(request).get(requestTimeout, TimeUnit.SECONDS));
    }

    public List<V> lookup(List<V> keys) throws Exception {
        var lookupSchema = messageConverter.getTableSchema().toLookup();
        var request = new LookupRowsRequest(tableProperties.getPath(), lookupSchema);
        for (var key : keys) {
            request.addFilter(messageConverter.getFilterValues(key));
        }
        request.addLookupColumns(messageConverter.getLookupColumns());

        var rowSet = retryTemplate.execute(c -> client.lookupRows(request).get(requestTimeout, TimeUnit.SECONDS));
        var result = new ArrayList<V>();
        for (var row : rowSet.getRows()) {
            var rowValues = new ArrayList<>(row.getValues());
            result.add(messageConverter.deserialize(rowValues));
        }
        return result;
    }

    public List<V> select(List<K> keys) throws Exception {

        var result = new ArrayList<V>();
        if (keys.isEmpty()) {
            return result;
        }

        String columns = String.join(", ", messageConverter.getLookupColumns());
        String selectKeys = String.join(", ", messageConverter.getSelectColumns());
        String selectValues = keys
                .stream()
                .map(messageConverter::getSelectValues)
                .collect(Collectors.joining(", "));
        String query = String.format(
                "%s from [%s] where (%s) in (%s)",
                columns,
                tableProperties.getPath(),
                selectKeys,
                selectValues
        );

        var rowSet = retryTemplate.execute(c -> client.selectRows(query).get(requestTimeout, TimeUnit.SECONDS));
        for (var row : rowSet.getRows()) {
            var rowValues = new ArrayList<>(row.getValues());
            result.add(messageConverter.deserialize(rowValues));
        }
        return result;
    }

    public void replace(List<V> rowsToDelete, List<V> rowsToInsert) throws Exception {
        try (var transaction = client.startTransaction(transactionOptions).get(requestTimeout, TimeUnit.SECONDS)) {
            replace(rowsToDelete, rowsToInsert, transaction);
            commitTransaction(transaction);
        }
    }

    public void replace(List<V> rowsToDelete, List<V> rowsToInsert, ApiServiceTransaction transaction) throws Exception {
        final var deleteRequest = new ModifyRowsRequest(tableProperties.getPath(), messageConverter.getTableSchema());
        for (var key : rowsToDelete) {
            deleteRequest.addDelete(messageConverter.getFilterValues(key));
        }
        final var insertRequest = getInsertRequest(rowsToInsert);

        retryTemplate.execute(c -> {
            if (!deleteRequest.getRows().isEmpty()) {
                transaction.modifyRows(deleteRequest).get(requestTimeout, TimeUnit.SECONDS);
            }
            if (!insertRequest.getRows().isEmpty()) {
                transaction.modifyRows(insertRequest).get(requestTimeout, TimeUnit.SECONDS);
            }
            return null;
        });
    }

    @Override
    public void close() {
        this.client.close();
    }

    private ModifyRowsRequest getInsertRequest(List<V> messages) {
        var request = new ModifyRowsRequest(tableProperties.getPath(), messageConverter.getTableSchema());
        for (var message : messages) {
            request.addInsert(messageConverter.serialize(message));
        }
        return request;
    }
}
