package ru.yandex.travel.cpa.data_processing.flow.yt;

import io.micrometer.core.instrument.Counter;
import io.netty.channel.nio.NioEventLoopGroup;
import org.springframework.retry.RetryCallback;
import org.springframework.retry.RetryContext;
import org.springframework.retry.backoff.ExponentialBackOffPolicy;
import org.springframework.retry.listener.RetryListenerSupport;
import org.springframework.retry.policy.SimpleRetryPolicy;
import org.springframework.retry.support.RetryTemplate;

import ru.yandex.yt.ytclient.bus.DefaultBusConnector;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.rpc.RpcCredentials;
import ru.yandex.yt.ytclient.rpc.RpcOptions;

public class SyncYtClientFactory {
    private final YtClient client;
    private final YtConnectionProperties connectionProperties;
    private final RetryTemplate retryTemplate;


    public SyncYtClientFactory(YtConnectionProperties connectionProperties, Counter errorsCounter) {
        this.connectionProperties = connectionProperties;

        var busConnector = new DefaultBusConnector(new NioEventLoopGroup(), true);
        this.client = new YtClient(
                busConnector,
                connectionProperties.getProxyName(),
                new RpcCredentials(
                        connectionProperties.getUser(),
                        connectionProperties.getToken()
                ),
                new RpcOptions()
                        .setGlobalTimeout(connectionProperties.getGlobalTimeout())
                        .setFailoverTimeout(connectionProperties.getFailoverTimeout())
                        .setPingTimeout(connectionProperties.getPingTimeout())
        );

        retryTemplate = new RetryTemplate();

        ExponentialBackOffPolicy backOffPolicy = new ExponentialBackOffPolicy();
        backOffPolicy.setInitialInterval(connectionProperties.getRetryInitialInterval().toMillis());
        backOffPolicy.setMaxInterval(connectionProperties.getRetryMaxInterval().toMillis());
        backOffPolicy.setMultiplier(connectionProperties.getRetryMultiplier());
        retryTemplate.setBackOffPolicy(backOffPolicy);

        SimpleRetryPolicy retryPolicy = new SimpleRetryPolicy();
        retryPolicy.setMaxAttempts(connectionProperties.getRetryAttempts());
        retryTemplate.setRetryPolicy(retryPolicy);

        retryTemplate.registerListener(new RetryListenerSupport() {
            @Override
            public <T, E extends Throwable> void onError(
                    RetryContext context, RetryCallback<T, E> callback, Throwable throwable
            ) {
                errorsCounter.increment();
            }
        });
    }

    public <K, V> SyncYtClient<K, V> newSyncYtClient(YtTableProperties tableProperties, YtMessageConverter<K, V> messageConverter) {
        return new SyncYtClient<>(client, connectionProperties, tableProperties, messageConverter, retryTemplate);
    }
}
