# -*- coding: utf-8 -*-

from collections import OrderedDict, namedtuple

from travel.cpa.data_processing.lib.label_converter import LABEL_CONVERTERS
from travel.cpa.data_processing.lib.order_processor import (
    AviaOrderProcessor, BusesOrderProcessor, HotelsOrderProcessor, TrainOrderProcessor, GenericOrderProcessor,
    SuburbanOrderProcessor, ToursOrderProcessor
)
from travel.cpa.lib.order_snapshot import BusesOrder, TrainOrder, SuburbanOrder, ToursOrder
from travel.library.python.schematized import Schematized
from travel.library.python.schematized.fields import Any, Boolean, Float, Int64, UInt64, String


class Order(Schematized):
    __fields__ = OrderedDict([
        ('travel_order_id', String()),
        ('partner_name', String()),
        ('partner_order_id', String()),
        ('status', String()),
        ('currency_code', String()),
        ('order_amount', Float()),
        ('order_amount_rub', Float()),
        ('profit_amount', Float(optional=True)),
        ('profit_amount_rub', Float(optional=True)),
        ('created_at', UInt64()),
        ('updated_at', UInt64()),
        ('category', String()),

        ('first_seen_at', UInt64()),
        ('is_suspicious', Boolean()),
        ('status_explanation', Any()),
        ('suspicious_explanation', Any(optional=True)),
        ('label', String(optional=True, empty_is_none=True)),
    ])


class AviaOrder(Order):
    __fields__ = OrderedDict([
        ('source', String()),
        ('partner_id', UInt64()),
        ('billing_order_id', UInt64()),
        ('origin', String(optional=True)),
        ('trip_type', String(optional=True)),
        ('destination', String(optional=True)),
        ('date_forward', String(optional=True)),
        ('date_backward', String(optional=True)),
        ('profit_amount_ex_tax', Float(default=0.0)),
        (
            'service_fee',
            Float(optional=True, description='Сумма, за которую комиссию мы не получаем, входит в order_amount')
        ),
        ('partner_profit', Float(optional=True, description='Сумма нашей выручки по расчетам партнера')),
        ('pnr', String(optional=True)),
        ('unisearchquery', String(optional=True))
    ])


class AviaOrderWithEncodedLabel(AviaOrder):
    __fields__ = OrderedDict([
        ('snapshot_source', String(optional=True)),
    ])


class AviaOrderWithDecodedLabel(AviaOrder, LABEL_CONVERTERS['avia'].fields_cls):
    __fields__ = OrderedDict(has_label=Boolean())


class BusesOrderWithEncodedLabel(BusesOrder, Order):
    pass


class HotelsOrder(Order):
    __fields__ = OrderedDict([
        ('order_amount_payable', Float(optional=True)),
        ('profit_amount_ex_tax', Float(default=0.0)),
        ('partner_status', String(optional=True, empty_is_none=True)),
        ('check_in', String()),
        ('check_out', String()),
        ('hotel_name', String(optional=True)),
        ('hotel_country', String(optional=True, empty_is_none=True)),
        ('hotel_city', String(optional=True, empty_is_none=True)),
        ('partner_name', String()),
        ('hotel_type', String(optional=True, empty_is_none=True)),
        ('promo_codes', String(optional=True)),
        ('promo_actions', String(optional=True)),
        ('discount_amount', Float(optional=True)),
        ('promocode_discount_amount', Float(optional=True)),
        ('strike_through_discount_amount', Float(optional=True)),
        ('privetmir_oct2020_eligible', Boolean(optional=True)),
        ('use_deferred_payment', Boolean(optional=True)),
        ('post_pay_eligible', Boolean(default=False)),
        ('post_pay_used', Boolean(default=False)),
        ('amount_received_from_user', Float(optional=True)),
        ('refund_reason', String(optional=True)),
        ('deferred_payment_eligibility', String(optional=True)),
        ('initial_payment_amount', Float(optional=True)),
        ('penalty_amount_if_unpaid', Float(optional=True)),
        ('last_payment_scheduled_at', UInt64(optional=True)),
        ('fully_paid_at', UInt64(optional=True)),
        ('partner_hotel_id', String(optional=True)),

        ('yandex_plus_mode', String(optional=True)),
        ('yandex_plus_withdraw_points', Int64(optional=True)),
        ('yandex_plus_topup_points', Int64(optional=True)),
        ('yandex_plus_topup_date', UInt64(optional=True)),
        ('yandex_plus_user_balance', Int64(optional=True)),

        ('white_label_points_type', String(optional=True)),
        ('white_label_points_amount', Int64(optional=True)),
        ('white_label_customer_number', String(optional=True, hidden=True)),
    ])


class HotelsOrderWithEncodedLabel(HotelsOrder):
    __fields__ = OrderedDict([
        ('snapshot_source', String(optional=True)),
    ])


class HotelsOrderWithDecodedLabel(HotelsOrder, LABEL_CONVERTERS['hotels'].fields_cls):
    __fields__ = OrderedDict(has_label=Boolean())


class GenericOrderWithEncodedLabel(Order):
    __fields__ = OrderedDict([
        ('display_type', String())
    ])


class GenericOrderWithDecodedLabel(GenericOrderWithEncodedLabel, LABEL_CONVERTERS['generic'].fields_cls):
    __fields__ = OrderedDict(has_label=Boolean())


class TrainOrderWithEncodedLabel(TrainOrder, Order):
    __fields__ = OrderedDict([
        ('item_state', String(optional=True))
    ])


class TrainOrderWithDecodedLabel(TrainOrderWithEncodedLabel, LABEL_CONVERTERS['train'].fields_cls):
    __fields__ = OrderedDict(has_label=Boolean())


class BusesOrderWithDecodedLabel(BusesOrderWithEncodedLabel, LABEL_CONVERTERS['buses'].fields_cls):
    __fields__ = OrderedDict(has_label=Boolean())


class SuburbanOrderWithEncodedLabel(SuburbanOrder, Order):
    __fields__ = OrderedDict([
        ('item_state', String(optional=True))
    ])


class SuburbanOrderWithDecodedLabel(SuburbanOrderWithEncodedLabel, LABEL_CONVERTERS['suburban'].fields_cls):
    __fields__ = OrderedDict(has_label=Boolean())


class ToursOrderWithEncodedLabel(ToursOrder, Order):
    __fields__ = OrderedDict()


class ToursOrderWithDecodedLabel(ToursOrderWithEncodedLabel, LABEL_CONVERTERS['tours'].fields_cls):
    __fields__ = OrderedDict(has_label=Boolean())


CategoryConfig = namedtuple(
    'CategoryConfig',
    ['order_with_encoded_label_cls', 'order_with_decoded_label_cls', 'order_processor_cls'],
)


CATEGORY_CONFIGS = {
    'avia': CategoryConfig(AviaOrderWithEncodedLabel, AviaOrderWithDecodedLabel, AviaOrderProcessor),
    'buses': CategoryConfig(BusesOrderWithEncodedLabel, BusesOrderWithDecodedLabel, BusesOrderProcessor),
    'generic': CategoryConfig(GenericOrderWithEncodedLabel, GenericOrderWithDecodedLabel, GenericOrderProcessor),
    'hotels': CategoryConfig(HotelsOrderWithEncodedLabel, HotelsOrderWithDecodedLabel, HotelsOrderProcessor),
    'train': CategoryConfig(TrainOrderWithEncodedLabel, TrainOrderWithDecodedLabel, TrainOrderProcessor),
    'suburban': CategoryConfig(SuburbanOrderWithEncodedLabel, SuburbanOrderWithDecodedLabel, SuburbanOrderProcessor),
    'tours': CategoryConfig(ToursOrderWithEncodedLabel, ToursOrderWithDecodedLabel, ToursOrderProcessor),
}
