# -*- coding: utf-8 -*-

from copy import deepcopy
from datetime import date, datetime
from typing import Any

import pytz
from dateutil.relativedelta import relativedelta

from travel.cpa.lib.lib_datetime import parse_datetime_iso, timestamp, timestamp_to_str
from travel.library.python.currency_converter import CurrencyConverter
from travel.library.python.schematized import Schematized

FIELDS_TO_IGNORE = {'snapshot_source', 'status', 'partner_status', 'updated_at'}

HOUR = 60 * 60

TS_2019_01_01 = timestamp(datetime(year=2019, month=1, day=1))
TS_2019_10_01 = timestamp(datetime(year=2019, month=10, day=1))
TS_2019_11_01 = timestamp(datetime(year=2019, month=11, day=1))


class OrderProcessor:
    PARTNER_NAME = None

    def __init__(self, currency_converter: CurrencyConverter):
        self.currency_converter = currency_converter

    def get_fx_rate(self, order: Schematized) -> float:
        # This method for RUB values only. If you need another currency, implement it at partner processor level
        if order.currency_code != 'RUB':
            raise ValueError('Non RUB value for converting')
        return 1

    @staticmethod
    def is_status_change_ok(status_from: str, status_to: str):
        if status_from is None:
            return True
        if status_from == 'pending' and status_to in {'confirmed', 'refunded'}:
            return True
        if status_from == 'unpaid' and status_to in {'confirmed', 'cancelled'}:
            return True
        return False

    @staticmethod
    def get_change_explanation(
        prev_snapshot: Schematized,
        next_snapshot: Schematized,
        fields_to_ignore: set[str],
    ) -> [(str, str, Any, Any)]:
        if prev_snapshot is None:
            return
        for key, prev_value, next_value in Schematized.diff(prev_snapshot, next_snapshot):
            if key in fields_to_ignore:
                continue
            if prev_value is None:
                continue
            if prev_value == 0:
                continue
            if key == 'created_at':
                prev_value = timestamp_to_str(prev_value)
                next_value = timestamp_to_str(next_value)
            yield timestamp_to_str(next_snapshot.updated_at), key, prev_value, next_value

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = deepcopy(snapshots[-1])
        order.is_suspicious = False

        created_at = order.created_at
        updated_at = order.updated_at
        profit_amount = None
        first_seen_at = order.updated_at
        status_explanation = list()
        suspicious_explanation = list()
        prev_snapshot = None
        current_status = None
        was_confirmed = False

        for snapshot in snapshots:
            profit_amount = snapshot.profit_amount
            created_at = min(created_at, snapshot.created_at)
            updated_at = max(updated_at, snapshot.updated_at)
            first_seen_at = min(first_seen_at, snapshot.updated_at)

            new_status = snapshot.status

            if was_confirmed:
                suspicious_explanation.extend(self.get_change_explanation(prev_snapshot, snapshot, FIELDS_TO_IGNORE))

            prev_snapshot = snapshot
            if snapshot.status == current_status:
                continue
            status_explanation.append((timestamp_to_str(snapshot.updated_at), current_status, new_status))
            if not self.is_status_change_ok(current_status, new_status):
                suspicious_explanation.append((
                    timestamp_to_str(snapshot.updated_at), 'status', current_status, new_status
                ))
            current_status = new_status

            if current_status == 'confirmed':
                was_confirmed = True

        order.profit_amount = profit_amount
        order.created_at = created_at
        order.updated_at = updated_at
        order.first_seen_at = first_seen_at
        order.status_explanation = status_explanation
        order.suspicious_explanation = suspicious_explanation
        order.is_suspicious = len(suspicious_explanation) > 0

        rate = self.get_fx_rate(order)
        if rate is not None:
            order_amount = order.order_amount
            if order_amount is None:
                order_amount = 0.0
                order.order_amount = order_amount
            order.order_amount_rub = order_amount * rate
            if profit_amount is not None:
                order.profit_amount_rub = order.profit_amount * rate
        else:
            order.order_amount_rub = None

        # booking sends us strange labels, this will fix it
        if order.label == 'None':
            order.label = None

        return order


#
# avia processors
#
class AviaOrderProcessor(OrderProcessor):
    def get_fx_rate(self, order: Schematized) -> float:
        if order.currency_code == 'RUB':
            return 1

        order_creation_date = datetime.fromtimestamp(order.created_at, tz=pytz.utc).date()
        rate = self.currency_converter.get_rate(order.currency_code, order_creation_date)
        if rate is None:
            raise ValueError('Cannot convert currency %s for date %s' % (order.currency_code, order_creation_date))
        return rate

    def zero_profit_for_failed_labels(self, order):
        """RASPTICKETS-21874 сгенерировали невалидные повторяющиеся маркеры.
        Решили занулить профит и сумму заказа, чтобы не ломать статистику, которая строится на уникальности маркера"""
        if not order.label or order.label.startswith('generate_failed') or order.label.startswith('generatefailed'):
            suffix = order.partner_name + order.partner_order_id
            if not order.label or not order.label.endswith(suffix):
                order.label = str(order.label) + suffix
            order.order_amount = 0.0
            order.order_amount_rub = 0.0
            order.profit_amount = 0.0
            order.profit_amount_rub = 0.0
            order.profit_amount_ex_tax = 0.0

    def process(self, snapshots):
        order = super(AviaOrderProcessor, self).process(snapshots)
        self.zero_profit_for_failed_labels(order)
        return order


class AeroflotOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'aeroflot'
    PROFIT_AMOUNT_EX_TAX = 179.17

    def process(self, snapshots):
        order = super(AeroflotOrderProcessor, self).process(snapshots)

        order.profit_amount_rub = order.profit_amount
        order.profit_amount_ex_tax = self.PROFIT_AMOUNT_EX_TAX
        return order


class AgentOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'agent'


class AviakassaOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'aviakassa'


class AviaoperatorOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'aviaoperator'


class AzimuthOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'azimuth'


class BiletdvOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'biletdv'


class BiletikaeroagOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'biletikaeroag'


class BiletinetOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'biletinet'


class BiletixOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'biletix'


class BookandtripOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'booktripruag'


class CitytravelOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'citytravel'


class Citytravel1OrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'citytravel1'


class ClickaviaOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'clickavia'


class ExpressaviaOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'expressavia'


class GogateOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'gogate'


class KiwiOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'kiwi'

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = super(KiwiOrderProcessor, self).process(snapshots)

        order.profit_amount = order.order_amount * .03
        order.profit_amount_rub = order.order_amount_rub * .03
        order.profit_amount_ex_tax = order.profit_amount_rub
        return order


class KupibiletOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'kupibilet'

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = super(KupibiletOrderProcessor, self).process(snapshots)

        order.profit_amount = order.order_amount * .025
        order.profit_amount_rub = order.order_amount_rub * .025
        order.profit_amount_ex_tax = order.profit_amount_rub / 1.2

        return order


class MegotravelOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'megotravel'


class NabortuOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'nabortu'


class NebotravelOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'nebotravel'


class OnetwotripruProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'onetwotripru'


class OzonOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'ozon'


class PobedaOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'pobeda'

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = super(PobedaOrderProcessor, self).process(snapshots)

        order.profit_amount = order.order_amount * 0.01
        order.profit_amount_rub = order.order_amount_rub * 0.01
        order.profit_amount_ex_tax = order.profit_amount_rub / 1.2

        return order


class RedwingsOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'redwings'

    def process(self, snapshots):
        order = super(RedwingsOrderProcessor, self).process(snapshots)

        rate = self.get_fx_rate(order)
        order.profit_amount = order.order_amount * 1.75 / 100.
        order.profit_amount_rub = order.profit_amount * rate
        order.profit_amount_ex_tax = order.profit_amount_rub / 1.2
        return order


class RuslineOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'rusline'


class SmartaviaOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'smartavia'

    def process(self, snapshots):
        order = super(SmartaviaOrderProcessor, self).process(snapshots)

        rate = self.get_fx_rate(order)
        order.profit_amount = order.order_amount * 1.75 / 100.
        order.profit_amount_rub = order.profit_amount * rate
        order.profit_amount_ex_tax = order.profit_amount_rub / 1.2
        return order


class SSevenOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 's_seven'

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = super(SSevenOrderProcessor, self).process(snapshots)

        rate = self.get_fx_rate(order)
        order.profit_amount = (order.order_amount - (order.service_fee or 0)) * 1.2 * 1.75 / 100.
        order.profit_amount_rub = order.profit_amount * rate
        order.profit_amount_ex_tax = order.profit_amount_rub / 1.2

        self.zero_profit_for_failed_labels(order)
        return order


class SuperkassaOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'superkassa'


class SupersaverOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'supersaver'


class SvyaznoyOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'svyaznoy'


class TinkoffOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'tinkoff'


class TicketsruOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'ticketsru'


class TripruOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'trip_ru'


class TutuOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'tutu'

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = super(TutuOrderProcessor, self).process(snapshots)

        # https://st.yandex-team.ru/RASPTICKETS-22950
        if order.unisearchquery == 'brand':
            order.profit_amount = 0.0
            order.profit_amount_rub = 0.0
            order.profit_amount_ex_tax = 0.0
        else:
            order.profit_amount = order.order_amount * .03
            order.profit_amount_rub = order.order_amount_rub * .03
            order.profit_amount_ex_tax = order.profit_amount_rub / 1.2

        self.zero_profit_for_failed_labels(order)
        return order


class UzairwaysOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'uzairways'


class UtairOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'utair'

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = super(UtairOrderProcessor, self).process(snapshots)

        if order.order_amount <= 7000:
            profit_amount = 120.0
        elif order.order_amount <= 10000:
            profit_amount = 150.0
        else:
            profit_amount = 170.0

        order.profit_amount = profit_amount
        order.profit_amount_rub = profit_amount
        order.profit_amount_ex_tax = order.profit_amount_rub / 1.2

        self.zero_profit_for_failed_labels(order)
        return order


class AnywayanydayOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'anywayanyday'


class TripcomOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'tripcom'


class UralAirlinesOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'uralairlines'

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = super(UralAirlinesOrderProcessor, self).process(snapshots)

        order.profit_amount = order.order_amount * 0.015
        order.profit_amount_rub = order.order_amount_rub * 0.015
        order.profit_amount_ex_tax = order.profit_amount_rub
        return order


class FlyoneOrderProcessor(AviaOrderProcessor):
    PARTNER_NAME = 'flyone'


# hotels processors


class HotelsOrderProcessor(OrderProcessor):

    def process(self, snapshots: list[Schematized]) -> Schematized:
        hotel_city = None
        for snapshot in snapshots:
            # workaround for history import issues
            if snapshot.snapshot_source is None:
                snapshot.created_at = snapshot.created_at - 3 * HOUR
            # workaround for hotel_city partial absence
            hotel_city = snapshot.hotel_city
            if hotel_city is not None:
                hotel_city = hotel_city.strip()
        order = super(HotelsOrderProcessor, self).process(snapshots)
        order.hotel_city = hotel_city
        order.profit_amount_ex_tax = order.profit_amount_rub
        return order


class BookingProcessor(HotelsOrderProcessor):
    PARTNER_NAME = 'booking'

    def get_fx_rate(self, order: Schematized) -> float:
        check_out_date = parse_datetime_iso(order.check_out).date()
        fx_date = check_out_date.replace(day=5) + relativedelta(months=+2)
        fx_date = min(fx_date, date.today())
        return self.currency_converter.get_rate(order.currency_code, fx_date)


class Hotels101Processor(HotelsOrderProcessor):
    PARTNER_NAME = 'hotels101'

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = super(Hotels101Processor, self).process(snapshots)
        # we have commission starting from this date
        # details at HOTELS-3825, HOTELS-3860
        if order.created_at < TS_2019_01_01:
            order.profit_amount = .0
            order.profit_amount_rub = .0
        # taxes HOTELS-3860
        order.profit_amount_ex_tax = order.profit_amount_rub * 0.8
        return order


class HotelsCombinedProcessor(HotelsOrderProcessor):
    PARTNER_NAME = 'hotelscombined'

    def get_fx_rate(self, order: Schematized) -> float:
        check_out_date = parse_datetime_iso(order.check_out).date()
        fx_date = check_out_date.replace(day=5) + relativedelta(months=+1)
        fx_date = min(fx_date, date.today())
        return self.currency_converter.get_rate(order.currency_code, fx_date)

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = super(HotelsCombinedProcessor, self).process(snapshots)
        # convert Australian tz to UTC
        # TODO: fix this at collector level and convert snapshots
        order.created_at -= 11 * HOUR
        return order


class LevelTravelProcessor(HotelsOrderProcessor):
    PARTNER_NAME = 'leveltravel'


class OstrovokProcessor(HotelsOrderProcessor):
    PARTNER_NAME = 'ostrovok'

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = super(OstrovokProcessor, self).process(snapshots)
        # temporary disable this logic
        # TODO: figure out what to do with 2018 year orders
        # # taxes HOTELS-3860
        # if order.check_out < TS_2019_10_01:
        #     order.profit_amount_ex_tax = order.profit_amount_rub * 0.82
        # elif order.check_out < TS_2019_11_01:
        #     order.profit_amount = order.profit_amount / 0.82
        #     order.profit_amount_rub = order.profit_amount_rub / 0.82
        # else:
        #     order.profit_amount = order.profit_amount / 0.8
        #     order.profit_amount_rub = order.profit_amount_rub / 0.8
        order.profit_amount = order.order_amount * 0.114
        order.profit_amount_rub = order.profit_amount
        order.profit_amount_ex_tax = order.order_amount * 0.095
        return order


class ExpediaProcessor(HotelsOrderProcessor):
    PARTNER_NAME = 'expedia'

    # keep profit_amount calculation for old snapshots only
    # new snapshot profit_amount provided by order app
    def process(self, snapshots: list[Schematized]) -> Schematized:
        for snapshot in snapshots:
            if snapshot.profit_amount is None:
                snapshot.profit_amount = snapshot.order_amount * 0.085

        return super(ExpediaProcessor, self).process(snapshots)


class DolphinProcessor(HotelsOrderProcessor):
    PARTNER_NAME = 'dolphin'

    # keep profit_amount calculation for old snapshots only
    # new snapshot profit_amount provided by order app
    def process(self, snapshots: list[Schematized]) -> Schematized:
        for snapshot in snapshots:
            if snapshot.profit_amount is None:
                snapshot.profit_amount = snapshot.order_amount * 0.13

        return super(DolphinProcessor, self).process(snapshots)


class BronevikProcessor(HotelsOrderProcessor):
    PARTNER_NAME = 'bronevik'


class TravellineProcessor(HotelsOrderProcessor):
    PARTNER_NAME = 'travelline'


class BnovoProcessor(HotelsOrderProcessor):
    PARTNER_NAME = 'bnovo'


class TvilProcessor(HotelsOrderProcessor):
    PARTNER_NAME = 'tvil'

    def process(self, snapshots: list[Schematized]) -> Schematized:
        for snapshot in snapshots:
            # the quantity-based profit percent scheme can probably be implemented here later
            snapshot.profit_amount = round(snapshot.order_amount * 0.09, 2) if snapshot.partner_status != 'CANCELLED' else 0.0

        return super(TvilProcessor, self).process(snapshots)


# tours processors

class ToursOrderProcessor(OrderProcessor):
    pass


class LevelTravelWhitelabelProcessor(ToursOrderProcessor):
    PARTNER_NAME = 'leveltravel_whitelabel'


# generic processors

class GenericOrderProcessor(OrderProcessor):
    PARTNER_NAME = 'boy'


# train processors

class TrainOrderProcessor(OrderProcessor):
    pass


class ImProcessor(TrainOrderProcessor):
    PARTNER_NAME = 'im'


class ImBoyProcessor(TrainOrderProcessor):
    PARTNER_NAME = 'im_boy'


# suburban processors

class SuburbanOrderProcessor(OrderProcessor):
    pass


class SuburbanBoyProcessor(SuburbanOrderProcessor):
    PARTNER_NAME = 'suburban'


# Buses processors

class BusesOrderProcessor(OrderProcessor):
    pass


class AtlasbusBusesProcessor(BusesOrderProcessor):
    PARTNER_NAME = 'atlasbus'


class BusforBusesProcessor(BusesOrderProcessor):
    PARTNER_NAME = 'busfor'
    YANDEX_REVENUE = 0.67

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = super(BusforBusesProcessor, self).process(snapshots)
        if order.partner_commission:
            order.total_agency_fee_amount = round(order.partner_commission * self.YANDEX_REVENUE, 2)
            order.profit_amount = order.total_fee_amount + order.total_agency_fee_amount
        return order


class EcolinesBusesProcessor(BusesOrderProcessor):
    PARTNER_NAME = 'ecolines'


class EtrafficBusesProcessor(BusesOrderProcessor):
    PARTNER_NAME = 'etraffic'


class NoyBusesProcessor(BusesOrderProcessor):
    PARTNER_NAME = 'noy'


class OkBusesProcessor(BusesOrderProcessor):
    PARTNER_NAME = 'ok'


class RusetBusesProcessor(BusesOrderProcessor):
    PARTNER_NAME = 'ruset'


class SksBusesProcessor(BusesOrderProcessor):
    PARTNER_NAME = 'sks'


class UnitikiNewBusesProcessor(BusesOrderProcessor):
    PARTNER_NAME = 'unitiki-new'


class YugavtotransBusesProcessor(BusesOrderProcessor):
    PARTNER_NAME = 'yugavtotrans'


class BoyBusesProcessor(BusesOrderProcessor):
    PARTNER_NAME = 'bus-boy'

    def process(self, snapshots: list[Schematized]) -> Schematized:
        order = super(BoyBusesProcessor, self).process(snapshots)
        order.profit_amount = order.total_fee_amount + order.total_agency_fee_amount
        rate = self.get_fx_rate(order)
        if rate is not None and order.profit_amount is not None:
            order.profit_amount_rub = order.profit_amount * rate
        return order


PROCESSORS = {
    'avia': [
        AeroflotOrderProcessor,
        AgentOrderProcessor,
        AviakassaOrderProcessor,
        AviaoperatorOrderProcessor,
        AzimuthOrderProcessor,
        BiletdvOrderProcessor,
        BiletikaeroagOrderProcessor,
        BiletinetOrderProcessor,
        BiletixOrderProcessor,
        BookandtripOrderProcessor,
        ClickaviaOrderProcessor,
        CitytravelOrderProcessor,
        Citytravel1OrderProcessor,
        ExpressaviaOrderProcessor,
        FlyoneOrderProcessor,
        GogateOrderProcessor,
        KiwiOrderProcessor,
        KupibiletOrderProcessor,
        MegotravelOrderProcessor,
        NabortuOrderProcessor,
        NebotravelOrderProcessor,
        OnetwotripruProcessor,
        OzonOrderProcessor,
        PobedaOrderProcessor,
        RedwingsOrderProcessor,
        RuslineOrderProcessor,
        SmartaviaOrderProcessor,
        SSevenOrderProcessor,
        SuperkassaOrderProcessor,
        SupersaverOrderProcessor,
        SvyaznoyOrderProcessor,
        TinkoffOrderProcessor,
        TicketsruOrderProcessor,
        TripruOrderProcessor,
        TutuOrderProcessor,
        UzairwaysOrderProcessor,
        UralAirlinesOrderProcessor,
        UtairOrderProcessor,
        AnywayanydayOrderProcessor,
        TripcomOrderProcessor,
    ],
    'hotels': [
        BnovoProcessor,
        BronevikProcessor,
        BookingProcessor,
        ExpediaProcessor,
        DolphinProcessor,
        Hotels101Processor,
        HotelsCombinedProcessor,
        LevelTravelProcessor,
        OstrovokProcessor,
        TravellineProcessor,
        TvilProcessor,
    ],
    'train': [
        ImProcessor,
        ImBoyProcessor,
    ],
    'suburban': [
        SuburbanBoyProcessor,
    ],
    'buses': [
        AtlasbusBusesProcessor,
        BusforBusesProcessor,
        EcolinesBusesProcessor,
        EtrafficBusesProcessor,
        NoyBusesProcessor,
        OkBusesProcessor,
        RusetBusesProcessor,
        SksBusesProcessor,
        UnitikiNewBusesProcessor,
        YugavtotransBusesProcessor,
        BoyBusesProcessor,
    ],
    'generic': [
        GenericOrderProcessor,
    ],
    'tours': [
        LevelTravelWhitelabelProcessor,
    ],
}
