# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from argparse import ArgumentParser, Namespace
from typing import Any, Dict
import logging.config
import sys

from yt.wrapper import YtClient
import yt.wrapper as yt

from travel.cpa.data_processing.lib.batch_iterator import batch_iterator
from travel.cpa.lib.common import hide_secrets
from travel.cpa.lib.lib_logging import LOG_CONFIG, get_logger
from travel.cpa.lib.metered_task_executor import MeteredTaskExecutor
from travel.hotels.lib.python3.yql.yqllib import create_client, run_query, wait_results
from travel.library.python.tools import replace_args_from_env


LOG = get_logger('rebuild_orders')

BATCH_SIZE = 2000


class Executor(MeteredTaskExecutor):

    def __init__(self, options: Namespace, common_labels: Dict[str, Any]):
        super().__init__('rebuild_orders', options, common_labels)

        self.yt_client = YtClient(options.yt_proxy, options.yt_token)
        self.yql_client = create_client(db=options.yt_proxy, token=options.yql_token)

        self.hotels_table = yt.ypath_join(options.work_dir, 'hotels', 'orders')
        self.queue_table = yt.ypath_join(options.work_dir, 'order_queue_slow')

    def try_execute(self):
        temp = self.yt_client.create_temp_table()
        query = f'''
            INSERT INTO `{temp}` WITH TRUNCATE
            SELECT
                partner_name,
                partner_order_id,
            FROM `{self.hotels_table}`
            WHERE
                (partner_name == "booking" OR partner_name == "hotelscombined") AND
                check_out >= CAST(CurrentUtcDate() - DateTime::IntervalFromDays(90) AS String);
        '''
        request = run_query(self.yql_client, query)
        wait_results(request)

        for batch in batch_iterator(self.yt_client.read_table(temp, enable_read_parallel=True), BATCH_SIZE, None):
            self.yt_client.insert_rows(self.queue_table, batch)


def main():
    logging.config.dictConfig(LOG_CONFIG)

    parser = ArgumentParser()

    parser.add_argument('--work-dir', required=True)

    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)

    parser.add_argument('--yql-token', required=True)

    Executor.configure(parser)
    options = parser.parse_args(replace_args_from_env(sys.argv[1:]))

    displayed_options = hide_secrets(vars(options))
    LOG.info('Working with %r', displayed_options)

    common_labels = {'yt_proxy': options.yt_proxy}
    Executor.resolve_secrets(options, common_labels=common_labels)

    Executor(options, common_labels).execute()


if __name__ == '__main__':
    main()
