# -*- coding: utf-8 -*-
from argparse import ArgumentParser, Namespace
from collections import OrderedDict
from concurrent.futures import ThreadPoolExecutor, as_completed
from contextlib import contextmanager
from datetime import date, datetime, timedelta
from itertools import groupby
from typing import Any, Callable, Dict, Iterable, List, NamedTuple, Optional, Type
from time import monotonic
import json
import logging.config
# noinspection PyCompatibility
import sys

import yandex.type_info.typing as ti
from google.protobuf.message import Message
from yt.wrapper import YtClient
from yt.wrapper.operation_commands import Operation
from yt.wrapper.schema import ColumnSchema, TableSchema
import yt.wrapper as yt

from travel.cpa.data_processing.lib.label_converter import LABEL_CONVERTERS, LabelConverter
from travel.cpa.data_processing.lib.label_mapper import LABEL_MAPPERS, LabelMapper
from travel.cpa.data_processing.lib.order_data_model import CATEGORY_CONFIGS
from travel.cpa.data_processing.lib.order_processor import OrderProcessor, PROCESSORS
from travel.cpa.data_processing.lib.protobuf_utils import get_proto_hash
from travel.cpa.data_processing.update_orders_incremental.metrics import Counter, Metrics, Timer
from travel.cpa.data_processing.update_orders_incremental.order_killer import OrderKiller
from travel.cpa.data_processing.update_orders_incremental.order_notifier import OrderNotifier
from travel.cpa.lib.common import hide_secrets, with_retries
from travel.cpa.lib.lib_logging import LOG_CONFIG, get_logger
from travel.cpa.lib.metered_task_executor import MeteredTaskExecutor
from travel.cpa.lib.order_snapshot import OrderSnapshot
from travel.library.python.currency_converter import CurrencyConverter
from travel.library.python.time_interval import TimeInterval
from travel.library.python.tools import replace_args_from_env
# noinspection PyUnresolvedReferences
from travel.proto.cpa import commons_pb2 as commons_pb2
# noinspection PyUnresolvedReferences
from travel.proto.cpa import order_info_pb2 as order_info_pb2

LOG = get_logger('update_orders')

BATCH_SIZE = 2000

ORDER_LAG_BUCKETS = [
    '0m',
    '1m',
    '2m',
    '5m',
    '10m',
    '15m',
    '20m',
    '25m',
    '30m',
    '1h',
    '2h',
    '5h',
    '1d',
    '2d',
    '7d',
    '30d',
    '30d',
    '3650d',
]

TYPE_MAP = {
    'any': ti.Optional[ti.Yson],
    'boolean': ti.Optional[ti.Bool],
    'double': ti.Optional[ti.Double],
    'int64': ti.Optional[ti.Int64],
    'uint64': ti.Optional[ti.Uint64],
    'string': ti.Optional[ti.String],
}

STATUS_MAP = {
    'unpaid': commons_pb2.EOrderStatus.OS_UNPAID,
    'refunded': commons_pb2.EOrderStatus.OS_REFUNDED,
    'pending': commons_pb2.EOrderStatus.OS_PENDING,
    'paid': commons_pb2.EOrderStatus.OS_PAID,
    'confirmed': commons_pb2.EOrderStatus.OS_CONFIRMED,
    'cancelled': commons_pb2.EOrderStatus.OS_CANCELLED,
    'deleted': commons_pb2.EOrderStatus.OS_DELETED,
}

CURRENCY_MAP = {
    'AZN': commons_pb2.EOrderCurrency.OC_AZN,
    'EUR': commons_pb2.EOrderCurrency.OC_EUR,
    'RUB': commons_pb2.EOrderCurrency.OC_RUB,
    'USD': commons_pb2.EOrderCurrency.OC_USD,
    'CNY': commons_pb2.EOrderCurrency.OC_CNY,
    'TRY': commons_pb2.EOrderCurrency.OC_TRY,
    'KZT': commons_pb2.EOrderCurrency.OC_KZT,
    'UAH': commons_pb2.EOrderCurrency.OC_UAH,
    'CHF': commons_pb2.EOrderCurrency.OC_CHF,
    'BYN': commons_pb2.EOrderCurrency.OC_BYN,
    'KRW': commons_pb2.EOrderCurrency.OC_KRW,
    'PLN': commons_pb2.EOrderCurrency.OC_PLN,
    'GBP': commons_pb2.EOrderCurrency.OC_GBP,
    'AMD': commons_pb2.EOrderCurrency.OC_AMD,
    'ILS': commons_pb2.EOrderCurrency.OC_ILS,
    'UZS': commons_pb2.EOrderCurrency.OC_UZS,
    'KGS': commons_pb2.EOrderCurrency.OC_KGS,
}


class PartnerConfig(NamedTuple):
    order_with_encoded_label_cls: Type[OrderSnapshot]
    order_with_decoded_label_cls: Type[OrderSnapshot]
    processor: OrderProcessor
    category: str


class OrderKey(NamedTuple):
    partner_name: str
    partner_order_id: str


OrderDict = Dict[str, Any]


class OrderPurgatoryRow(NamedTuple):
    label: str
    partner_name: str
    partner_order_id: str
    updated_at: int

    def as_dict(self):
        return self._asdict()

    def key_dict(self):
        d = self._asdict()
        d.pop('updated_at')
        return d


class OrderUpdates(NamedTuple):
    orders_internal: Dict[str, List[OrderDict]]
    order_purgatory_insert: List[OrderPurgatoryRow]
    order_purgatory_delete: List[OrderPurgatoryRow]


class LabelKey(NamedTuple):
    order_category: str
    label_id: str


class LabelInfo(NamedTuple):
    label_proto: Message
    converter: LabelConverter
    mapper: Optional[LabelMapper]


def batch_iterator(items: Iterable[Any], batch_size: int, func: Optional[Callable[[Any], Any]]) -> Iterable[List[Any]]:
    buffer = list()
    for item in items:
        if func:
            item = func(item)
        buffer.append(item)
        if len(buffer) == batch_size:
            yield buffer
            buffer = list()
    if buffer:
        yield buffer


class Executor(MeteredTaskExecutor):
    __label_batch_size__ = 20
    __max_workers_count__ = 100

    __orders_table_name__ = 'orders'
    __orders_internal_table_name__ = 'orders_internal'

    def __init__(self, yt_client: YtClient, options: Namespace, common_labels: Dict[str, Any]):
        super().__init__('orders', options, common_labels)

        self.yt_client = yt_client
        self.max_working_time = options.max_working_time

        currency_converter = CurrencyConverter(base_url=options.stocks_url, start_date=date(2015, 1, 1))

        self.updated_at_max = self._get_updated_at_max()
        self.updated_at = dict()
        self.updated_at_with_label = dict()

        processor_map = dict()
        for category, processors in PROCESSORS.items():
            category_config = CATEGORY_CONFIGS[category]
            for processor_cls in processors:
                partner_config = PartnerConfig(
                    order_with_encoded_label_cls=category_config.order_with_encoded_label_cls,
                    order_with_decoded_label_cls=category_config.order_with_decoded_label_cls,
                    processor=processor_cls(currency_converter),
                    category=category,
                )
                processor_map[processor_cls.PARTNER_NAME] = partner_config
            self.updated_at[category] = list()
            self.updated_at_with_label[category] = list()
        self.processor_map: Dict[str, PartnerConfig] = processor_map

        self.labels_table = yt.ypath_join(options.workdir, 'labels')
        self.snapshots_table = yt.ypath_join(options.workdir, 'snapshots')
        self.order_queue_table = yt.ypath_join(options.workdir, 'order_queue')
        self.slow_queue_table = yt.ypath_join(options.workdir, 'order_queue_slow')
        self.order_purgatory_table = yt.ypath_join(options.workdir, 'order_purgatory')

        self.processed_regular_order_count = Counter('order_count', {'status': 'processed', 'type': 'regular'}, 0)
        self.processed_slow_order_count = Counter('order_count', {'status': 'processed', 'type': 'slow'}, 0)
        self.not_processed_regular_order_count = Counter(
            'order_count', {'status': 'not_processed', 'type': 'regular'}, 0
        )
        self.not_processed_slow_order_count = Counter('order_count', {'status': 'not_processed', 'type': 'slow'}, 0)
        self.metrics = Metrics(self.solomon_client.send)
        self.metrics.add_all(
            self.processed_regular_order_count,
            self.processed_slow_order_count,
            self.not_processed_regular_order_count,
            self.not_processed_slow_order_count,
        )

        today = datetime.utcnow().date()
        yesterday = today - timedelta(days=1)
        for code in currency_converter.CURRENCY_INDEX.keys():
            rate = currency_converter.get_rate(code, today)
            metric = Counter('currency_rate', {'code': code}, rate)
            self.metrics.add(metric)

            rate = currency_converter.get_rate(code, yesterday)
            metric = Counter('currency_rate_yesterday', {'code': code}, rate)
            self.metrics.add(metric)

        self.executor = ThreadPoolExecutor(max_workers=self.__max_workers_count__)

        self.notification_mapper = {
            'hotels': self._prepare_notification_hotels,
            'avia': self._prepare_notification_avia,
        }

        self.order_notifier = OrderNotifier(
            url=self.options.lb_url,
            port=self.options.lb_port,
            topic=self.options.lb_topic,
            source_id=self.options.lb_source_id,
            token=self.options.lb_token,
        )

    def try_execute(self):
        start = monotonic()

        self._mount_order_tables()
        self._migrate_schema()
        OrderKiller(
            yt_client=self.yt_client,
            workdir=self.options.workdir,
            categories=CATEGORY_CONFIGS.keys(),
        ).execute()

        self._check_table_exists(self.snapshots_table)
        self._check_table_exists(self.order_queue_table)

        if self._process_queue(start, self.order_queue_table, self.processed_regular_order_count, True):
            self._process_queue(start, self.slow_queue_table, self.processed_slow_order_count, False)

        if not self.options.skip_dump_tables:
            self._dump_tables()

        if not self.options.skip_set_medium_attributes:
            self._set_medium_attributes()

        self.not_processed_regular_order_count.value = self._get_row_count(self.order_queue_table)
        self.not_processed_slow_order_count.value = self._get_row_count(self.slow_queue_table)
        self._calc_order_lag_metrics()
        self.metrics.send()

    def _mount_order_tables(self) -> None:
        for category in PROCESSORS.keys():
            table_path = self._get_orders_internal_path(category)
            self.yt_client.mount_table(table_path)

    def _get_orders_internal_path(self, category: str) -> str:
        return yt.ypath_join(self.options.workdir, category, self.__orders_internal_table_name__)

    def _migrate_schema(self) -> None:
        for category, category_config in CATEGORY_CONFIGS.items():
            order_cls = category_config.order_with_decoded_label_cls or category_config.order_with_encoded_label_cls
            expected_schema = order_cls().get_yt_schema()
            expected_schema = self._get_table_schema(expected_schema, sort_by=['partner_name', 'partner_order_id'])
            table_path = self._get_orders_internal_path(category)
            actual_schema = self.yt_client.get_attribute(table_path, 'schema')
            actual_schema = TableSchema.from_yson_type(actual_schema)
            merged_schema = self._merge_schema(actual_schema, expected_schema)
            if actual_schema == merged_schema:
                continue
            LOG.info(f'Schema changed for {table_path}. Start migration')
            with_retries(self.yt_client.unmount_table)(table_path, sync=True)
            with_retries(self.yt_client.alter_table)(table_path, merged_schema)
            with_retries(self.yt_client.mount_table)(table_path)
            LOG.info('Migration finished')

    @staticmethod
    def _get_table_schema(schema_dict: Dict[str, str], sort_by: List[str]) -> TableSchema:
        table_schema = TableSchema()
        for field_name, field_type in schema_dict.items():
            column = ColumnSchema(field_name, TYPE_MAP[field_type])
            if field_name in sort_by:
                column.sort_order = 'ascending'
            table_schema.add_column(column)
        return table_schema

    @staticmethod
    def _merge_schema(old: TableSchema, new: TableSchema) -> TableSchema:
        columns = OrderedDict()
        for column in old.columns:
            columns[column.name] = column
        for column in new.columns:
            columns[column.name] = column
        merged_schema = TableSchema()
        merged_schema.strict = old.strict
        merged_schema.unique_keys = old.unique_keys
        for column in columns.values():
            merged_schema.add_column(column)
        return merged_schema

    def _check_table_exists(self, path: str) -> None:
        if not self.yt_client.exists(path):
            raise Exception(f'Table {path} not found, bye')

    def _process_queue(self, start: float, queue_table_path: str, order_counter: Counter, calc_lag: bool) -> bool:
        LOG.info(f'Processing {queue_table_path}')
        query = f'''
            [$row_index], partner_name, partner_order_id
            from [{queue_table_path}]
            where [$tablet_index] = 0
            limit {BATCH_SIZE}
        '''

        while True:
            cycle_start = monotonic()

            orders_to_process = with_retries(self.yt_client.select_rows)(query)
            orders_to_process = list(orders_to_process)

            if not orders_to_process:
                LOG.info('Nothing to process')
                return True

            LOG.info('Getting snapshots')
            snapshots = with_retries(self.yt_client.select_rows)(self._get_snapshots_query(orders_to_process))
            snapshots = [self._get_snapshot_data(s) for s in snapshots]
            labels_proto = self._get_labels_proto(snapshots)
            labels = self._get_labels_dict(labels_proto)
            orders = self._get_orders(snapshots, labels, calc_lag)

            for category, category_orders in orders.orders_internal.items():
                orders_table_path = self._get_orders_internal_path(category)
                LOG.info(f'Inserting {len(category_orders)} orders into {category} table')
                with_retries(self.yt_client.insert_rows)(orders_table_path, category_orders)

            LOG.info(f'Inserting {len(orders.order_purgatory_insert)} orders into {self.order_purgatory_table}')
            rows_to_insert = (row.as_dict() for row in orders.order_purgatory_insert)
            with_retries(self.yt_client.insert_rows)(self.order_purgatory_table, rows_to_insert)

            LOG.info(f'Deleting {len(orders.order_purgatory_delete)} orders from {self.order_purgatory_table}')
            rows_to_delete = (row.key_dict() for row in orders.order_purgatory_delete)
            with_retries(self.yt_client.delete_rows)(self.order_purgatory_table, rows_to_delete)

            if self.options.notify_order_change:
                self._notify_order_change(orders.orders_internal, labels_proto)

            trim_index = orders_to_process[-1]['$row_index'] + 1
            LOG.info(f'Trimming {queue_table_path} at {trim_index}')
            with_retries(self.yt_client.trim_rows)(queue_table_path, 0, trim_index)

            order_counter.value += len(orders_to_process)

            now = monotonic()
            ops = len(orders_to_process) / (now - cycle_start)
            LOG.info(f'{int(ops)} ops')

            if now - start >= self.max_working_time.interval:
                LOG.info('Max working time exceed')
                return False

    def _dump_tables(self):
        with self.yt_client.Transaction():
            with self._wait_operations() as operations:
                for category, category_config in CATEGORY_CONFIGS.items():
                    order_cls = (category_config.order_with_decoded_label_cls or
                                 category_config.order_with_encoded_label_cls)
                    src_path = self._get_orders_internal_path(category)
                    self.yt_client.unmount_table(src_path, sync=True)
                    operations.append(self._dump_category_orders(category, order_cls().get_hidden_field_names()))

            self._mount_order_tables()

            with self._wait_operations() as operations:
                for table_path in self._iter_order_dump_tables():
                    LOG.info(f'Sorting {table_path}')
                    operations.append(self.yt_client.run_sort(table_path, sort_by='created_at', sync=False))
            self._set_updated_at_max()

    def _iter_order_dump_tables(self):
        for category in PROCESSORS.keys():
            yield yt.ypath_join(self.options.dump_dir_public, category, self.__orders_table_name__)
            yield yt.ypath_join(self.options.dump_dir_private, category, self.__orders_table_name__)

    def _set_medium_attributes(self):
        LOG.info('Setting media attributes')
        attributes = {
            'default': {
                'replication_factor': 5,
                'data_parts_only': False,
            },
            'ssd_blobs': {
                'replication_factor': 3,
                'data_parts_only': False,
            }
        }
        for table_path in self._iter_order_dump_tables():
            self.yt_client.set(yt.ypath_join(table_path, '@media'), attributes, force=True)
            self.yt_client.set(yt.ypath_join(table_path, '@primary_medium'), 'ssd_blobs', force=True)

    def _get_snapshots_query(self, orders_to_process: List[OrderDict]) -> str:
        order_keys = set()
        for order_to_process in orders_to_process:
            order_key = self._get_order_key(order_to_process)
            order_keys.add(tuple(order_key))

        return f'''
            data from [{self.snapshots_table}]
            where (partner_name, partner_order_id) in ({', '.join(str(ok) for ok in order_keys)})
        '''

    @staticmethod
    def _get_snapshot_data(d: Dict[str, str]) -> OrderDict:
        return json.loads(d['data'])

    def _get_labels_proto(self, snapshots: List[OrderDict]) -> Dict[LabelKey, LabelInfo]:
        LOG.info('Getting labels')
        labels_to_lookup = set()
        generic_labels = set()
        for snapshot in snapshots:
            label_id = snapshot.get('label')
            if not label_id:
                continue
            partner_name = snapshot['partner_name']
            processor_config = self.processor_map.get(partner_name)
            if processor_config is None:
                LOG.warning(f'No suitable processor for {partner_name}')
                continue
            order_category = processor_config.category
            label_categories = [order_category]
            if order_category == 'generic':
                generic_labels.add(label_id)
                label_categories = CATEGORY_CONFIGS.keys()
            for label_category in label_categories:
                labels_to_lookup.add((label_category, label_id))
        labels_to_lookup = ({'category': category, 'label': label} for category, label in labels_to_lookup)
        labels = dict()

        LOG.info('Requesting labels')
        start = monotonic()
        label_futures = list()
        for batch in batch_iterator(labels_to_lookup, self.__label_batch_size__, None):
            future = self.executor.submit(with_retries(self.yt_client.lookup_rows), self.labels_table, batch)
            label_futures.append(future)

        raw_labels = list()
        for future in as_completed(label_futures):
            raw_labels.extend(future.result())

        now = monotonic()
        lps = len(raw_labels) / (now - start)
        LOG.info(f'{int(lps)} lps')

        LOG.info('Processing labels')
        generic_converter = LABEL_CONVERTERS['generic']
        for label in raw_labels:
            label_category = label['category']
            label_id = label['label']
            converter = LABEL_CONVERTERS[label_category]
            label_proto = converter.str_to_proto(label['data'])
            labels[LabelKey(label_category, label_id)] = LabelInfo(label_proto, converter, None)
            if label_id in generic_labels:
                mapper = LABEL_MAPPERS.get((label_category, 'generic'))
                labels[LabelKey('generic', label_id)] = LabelInfo(label_proto, generic_converter, mapper)
        LOG.info(f'Got {len(labels)} labels')
        return labels

    @staticmethod
    def _get_labels_dict(label_info: Dict[LabelKey, LabelInfo]) -> Dict[LabelKey, OrderDict]:
        labels = dict()
        for label_key, label_info in label_info.items():
            label_proto = label_info.label_proto
            if label_info.mapper is not None:
                label_proto = label_info.mapper.get_mapped_proto(label_proto)
            label_fields = label_info.converter.proto_to_dict(label_proto)
            labels[label_key] = label_fields
        return labels

    def _get_orders(
        self,
        snapshots: List[OrderDict],
        labels: Dict[LabelKey, OrderDict],
        calc_lag: bool,
    ) -> OrderUpdates:
        LOG.info('Building orders')
        orders_internal = dict()
        order_purgatory_insert = list()
        order_purgatory_delete = list()
        snapshots = sorted(snapshots, key=self._get_order_key)
        for order_key, order_snapshots in groupby(snapshots, key=self._get_order_key):
            partner_name = order_key.partner_name
            processor_config = self.processor_map.get(partner_name)
            if processor_config is None:
                LOG.warning(f'No suitable processor for {order_key}')
                continue
            try:
                order = self._get_order(order_snapshots, processor_config)
            except Exception as e:
                logging.error(f'Failed to process {order_key}\n{list(order_snapshots)}')
                raise e

            has_label = False
            order_with_decoded_label_cls = processor_config.order_with_decoded_label_cls
            if order_with_decoded_label_cls:
                has_label = False
                order_category = order['category']
                label_id = order['label']
                label = labels.get(LabelKey(order_category, label_id))
                if label is None:
                    if label_id:
                        order_purgatory_insert.append(self._get_order_purgatory_row(order))
                else:
                    order.update(label)
                    has_label = True
                    order_purgatory_delete.append(self._get_order_purgatory_row(order))
                order['has_label'] = has_label
                order = order_with_decoded_label_cls.from_dict(order, ignore_unknown=True, convert_type=True).as_dict()

            category_orders = orders_internal.setdefault(processor_config.category, list())
            category_orders.append(order)
            if calc_lag:
                updated_at = order['updated_at']
                current_updated_at_max = self.updated_at_max[processor_config.category]
                self.updated_at_max[processor_config.category] = max(current_updated_at_max, updated_at)
                self.updated_at[processor_config.category].append(updated_at)
                if has_label:
                    self.updated_at_with_label[processor_config.category].append(updated_at)
        return OrderUpdates(
            orders_internal=orders_internal,
            order_purgatory_insert=order_purgatory_insert,
            order_purgatory_delete=order_purgatory_delete,
        )

    @staticmethod
    def _get_order_key(d: Dict[str, Any]) -> OrderKey:
        return OrderKey(partner_name=d['partner_name'], partner_order_id=d['partner_order_id'])

    @staticmethod
    def _get_order(rows: [Dict[str, Any]], processor_config: PartnerConfig) -> OrderDict:
        raw_snapshots = sorted(rows, key=lambda x: x['updated_at'])
        order_cls = processor_config.order_with_encoded_label_cls
        snapshots = [order_cls.from_dict(s, ignore_unknown=True) for s in raw_snapshots]
        return processor_config.processor.process(snapshots).as_dict()

    @staticmethod
    def _get_order_purgatory_row(d: Dict[str, Any]) -> OrderPurgatoryRow:
        return OrderPurgatoryRow(
            label=d['label'],
            partner_name=d['partner_name'],
            partner_order_id=d['partner_order_id'],
            updated_at=d['updated_at'],
        )

    def _notify_order_change(self, orders: Dict[str, List[OrderDict]], labels: Dict[LabelKey, LabelInfo]) -> None:
        messages = list()
        LOG.info('Preparing notifications')
        for category, category_orders in orders.items():
            mapper = self.notification_mapper.get(category)
            if mapper is None:
                continue
            LOG.info(f'{len(category_orders)} {category} orders to notify')
            for order in category_orders:
                label_key = LabelKey(order['category'], order['label'])
                messages.append(mapper(order, labels.get(label_key)))
        if not messages:
            LOG.info('No messages to send')
            return
        LOG.info('Notifications prepared')
        self.order_notifier.notify(messages)

    @staticmethod
    def _prepare_notification_hotels(order: OrderDict, label: Optional[LabelInfo]) -> order_info_pb2.TGenericOrderInfo:
        message = order_info_pb2.TGenericOrderInfo()
        message.PartnerName = order['partner_name']
        message.PartnerOrderId = order['partner_order_id']
        message.OrderStatus = STATUS_MAP.get(order['status'], commons_pb2.EOrderStatus.OS_INVALID)
        message.Currency = CURRENCY_MAP.get(order['currency_code'], commons_pb2.EOrderCurrency.OC_INVALID)
        message.OrderAmount = order['order_amount']
        message.ProfitAmount = order['profit_amount']
        if label is not None:
            message.Label.LabelHotels.CopyFrom(label.label_proto)

        order_info_hotels = order_info_pb2.TOrderItemInfoHotels()
        order_info_hotels.PartnerName = message.PartnerName
        order_info_hotels.PartnerOrderId = message.PartnerOrderId
        order_info_hotels.OrderStatus = message.OrderStatus
        order_info_hotels.Currency = message.Currency
        order_info_hotels.OrderAmount = message.OrderAmount
        order_info_hotels.ProfitAmount = message.ProfitAmount
        order_info_hotels.CheckIn = order['check_in']
        order_info_hotels.CheckOut = order['check_out']

        order_info = message.OrderItems.add()
        order_info.OrderInfoHotels.CopyFrom(order_info_hotels)

        message.Hash = get_proto_hash(message)
        return message

    @staticmethod
    def _prepare_notification_avia(order: OrderDict, label: Optional[LabelInfo]) -> order_info_pb2.TGenericOrderInfo:
        message = order_info_pb2.TGenericOrderInfo()
        message.PartnerName = order['partner_name']
        message.PartnerOrderId = order['partner_order_id']
        message.OrderStatus = STATUS_MAP.get(order['status'], commons_pb2.EOrderStatus.OS_INVALID)
        message.Currency = CURRENCY_MAP.get(order['currency_code'], commons_pb2.EOrderCurrency.OC_INVALID)
        message.OrderAmount = order['order_amount']
        message.ProfitAmount = order.get('profit_amount') or 0.0
        if label is not None:
            message.Label.LabelAvia.CopyFrom(label.label_proto)

        order_info_avia = order_info_pb2.TOrderItemInfoAvia()
        order_info_avia.PartnerName = message.PartnerName
        order_info_avia.PartnerOrderId = message.PartnerOrderId
        order_info_avia.OrderStatus = message.OrderStatus
        order_info_avia.Currency = message.Currency
        order_info_avia.OrderAmount = message.OrderAmount
        order_info_avia.ProfitAmount = message.ProfitAmount
        order_info_avia.Origin = order.get('origin') or ''
        order_info_avia.Destination = order.get('destination') or ''
        order_info_avia.DateForward = order.get('date_forward') or ''
        order_info_avia.DateBackward = order.get('date_backward') or ''

        order_info = message.OrderItems.add()
        order_info.OrderInfoAvia.CopyFrom(order_info_avia)

        message.Hash = get_proto_hash(message)
        return message

    @contextmanager
    def _wait_operations(self):
        operations = list()
        yield operations
        for operation in operations:
            operation.wait()

    def _dump_category_orders(self, category: str, private_fields: set[str]) -> Operation:
        src_path = self._get_orders_internal_path(category)
        orders_public_path = yt.ypath_join(self.options.dump_dir_public, category, self.__orders_table_name__)
        orders_private_path = yt.ypath_join(self.options.dump_dir_private, category, self.__orders_table_name__)
        LOG.info(f'Dumping from {src_path} to {orders_public_path} and {orders_private_path}')
        LOG.info(f'{private_fields=}')
        self.yt_client.lock(src_path, mode='snapshot')

        src_schema = TableSchema.from_yson_type(self.yt_client.get_attribute(src_path, 'schema'))
        optimize_for = self.yt_client.get_attribute(src_path, 'optimize_for', default='lookup')

        self._prepare_table_to_dump(orders_public_path, src_schema, optimize_for, private_fields)
        self._prepare_table_to_dump(orders_private_path, src_schema, optimize_for)

        return self.yt_client.run_map(
            self._get_dump_mapper(private_fields),
            src_path, [orders_private_path, orders_public_path],
            sync=False,
            format=yt.YsonFormat(control_attributes_mode="iterator"),
        )

    @staticmethod
    def _get_dump_mapper(fields_to_skip: set[str]):
        @yt.with_context
        def dump_mapper(row: dict[str, Any], context):
            yield yt.create_table_switch(0)
            yield row
            yield yt.create_table_switch(1)
            yield {k: v for k, v in row.items() if k not in fields_to_skip}

        return dump_mapper

    def _prepare_table_to_dump(
        self,
        table_path: str,
        table_schema: TableSchema,
        optimize_for: str,
        fields_to_skip: set[str] = frozenset(),
    ):
        dst_columns = list()
        for column in table_schema.columns:
            if column.name in fields_to_skip:
                continue
            column.sort_order = None
            dst_columns.append(column)
        dst_schema = TableSchema(
            columns=dst_columns,
            strict=table_schema.strict,
        )
        attributes = {
            'optimize_for': optimize_for,
            'schema': dst_schema,
        }
        if self.yt_client.exists(table_path):
            LOG.info(f'Removing existing table {table_path}')
            self.yt_client.remove(table_path)

        self.yt_client.create('table', table_path, attributes=attributes, recursive=True)

    def _get_row_count(self, table_path: str) -> int:
        return self.yt_client.get_attribute(table_path, 'chunk_row_count')

    def _get_updated_at_max(self) -> Dict[str, int]:
        updated_at_max = dict()
        for category in PROCESSORS.keys():
            attribute_path = yt.ypath_join(
                self.options.dump_dir_public,
                category,
                self.__orders_table_name__,
                '@_updated_at_max',
            )
            value = 0
            try:
                value = self.yt_client.get(attribute_path)
            except Exception as e:
                LOG.warning(f'Failed to get attribute {attribute_path}, {e}')
            updated_at_max[category] = value
        return updated_at_max

    def _set_updated_at_max(self) -> None:
        for category, value in self.updated_at_max.items():
            attribute_path = yt.ypath_join(
                self.options.dump_dir_public,
                category,
                self.__orders_table_name__,
                '@_updated_at_max',
            )
            self.yt_client.set(attribute_path, value, force=True)

    def _calc_order_lag_metrics(self):
        now_ts = int(datetime.now().timestamp())
        LOG.info(f'now_ts = {now_ts}')
        buckets = ORDER_LAG_BUCKETS
        self._calc_order_lag_metrics_with_params(now_ts, 'all', buckets, self.updated_at)
        self._calc_order_lag_metrics_with_params(now_ts, 'with_label', buckets, self.updated_at_with_label)

    def _calc_order_lag_metrics_with_params(
        self,
        now_ts: int,
        status: str,
        buckets: List[str],
        measures: Dict[str, List[int]]
    ):
        for category, values in measures.items():
            base = Counter('order_lag', dict(category=category, status=status), 0)
            order_lag_timer = Timer(base, buckets)
            self.metrics.add(order_lag_timer)
            for value in values:
                lag = now_ts - value
                if lag < 0:
                    LOG.warning(f'Order from future detected {value}')
                    continue
                order_lag_timer.update(lag)


def main():
    logging.config.dictConfig(LOG_CONFIG)

    parser = ArgumentParser()

    parser.add_argument('--workdir', required=True)
    parser.add_argument('--dump-dir-public', required=True)
    parser.add_argument('--dump-dir-private', required=True)

    parser.add_argument('--stocks-url', default='http://stocks.yandex.net/xmlhist')

    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', default=None)

    parser.add_argument('--notify-order-change', action='store_true')
    parser.add_argument('--lb-url', default='logbroker.yandex.net')
    parser.add_argument('--lb-port', type=int, default=2135)
    parser.add_argument('--lb-token', default=None)
    parser.add_argument('--lb-topic', default=b'travel-cpa@dev--order-info')
    parser.add_argument('--lb-source-id', default='update_orders_incremental')

    parser.add_argument('--skip-dump-tables', action='store_true')
    parser.add_argument('--skip-set-medium-attributes', action='store_true')

    parser.add_argument('--max-working-time', type=TimeInterval, default=TimeInterval('1h'))

    Executor.configure(parser)
    options = parser.parse_args(replace_args_from_env(sys.argv[1:]))

    displayed_options = hide_secrets(vars(options))
    LOG.info('Working with %r', displayed_options)

    common_labels = {'yt_proxy': options.yt_proxy}
    Executor.resolve_secrets(options, common_labels=common_labels)

    yt_client = YtClient(options.yt_proxy, options.yt_token, config={'backend': 'rpc'})
    Executor(yt_client, options, common_labels).execute()


if __name__ == '__main__':
    main()
