# -*- coding: utf-8 -*-

from copy import deepcopy
from dataclasses import dataclass
from itertools import tee
from typing import Any, Callable, Dict, List, Union
import logging

from travel.library.python.time_interval import TimeInterval


@dataclass
class Bucket:
    value_range: range
    name: str
    item_count: int = 0


@dataclass
class Counter:
    sensor: str
    labels: Dict[str, str]
    value: Any


class Timer:

    def __init__(self, metric: Counter, buckets: List[Union[int, str]]):

        self.metric = metric
        self.buckets: List[Bucket] = list()

        a, b = tee(buckets)
        next(b, None)
        for p, n in zip(a, b):
            p = TimeInterval(p)
            n = TimeInterval(n)
            self.buckets.append(Bucket(
                value_range=range(p.interval, n.interval),
                name=f'{n.interval} ({n})',
            ))

    def update(self, value: int) -> None:
        for bucket in self.buckets:
            if value in bucket.value_range:
                bucket.item_count += 1
                return
        raise Exception(f'Value {value} is out of buckets range')

    def get_buckets(self) -> List[Counter]:
        metrics = list()
        for bucket in self.buckets:
            metric = deepcopy(self.metric)
            metric.labels['bin'] = bucket.name
            metric.value = bucket.item_count
            metrics.append(metric)
        return metrics


Metric = Union[Counter, Timer]


class Metrics:

    def __init__(self, send_callable: Callable):
        self.send_callable = send_callable
        self.metrics: List[Metric] = list()

    def add(self, metric: Metric) -> None:
        self.metrics.append(metric)

    def add_all(self, *metrics: Metric) -> None:
        self.metrics.extend(metrics)

    def send(self) -> None:
        for metric in self.metrics:
            if isinstance(metric, Timer):
                for bucket in metric.get_buckets():
                    logging.info(f'Sending {bucket}')
                    self.send_callable(bucket.sensor, bucket.value, **bucket.labels)
            else:
                logging.info(f'Sending {metric}')
                self.send_callable(metric.sensor, metric.value, **metric.labels)
