# -*- coding: utf-8 -*-

from typing import List
import logging

from yt.wrapper import YtClient
import yt.wrapper as yt

from travel.cpa.data_processing.lib.batch_iterator import batch_iterator


BATCH_SIZE = 5000


class OrderKiller:

    def __init__(self, yt_client: YtClient, workdir: str, categories: List[str]):
        self.yt_client = yt_client
        self.workdir = workdir
        self.categories = categories

    def execute(self) -> None:
        snapshot_keys_path = yt.ypath_join(self.workdir, 'inbox', 'snapshots_to_delete')
        snapshots_path = yt.ypath_join(self.workdir, 'snapshots')
        self._delete_rows(snapshot_keys_path, snapshots_path)

        for category in self.categories:
            order_keys_path = yt.ypath_join(self.workdir, category, 'inbox', 'orders_to_delete')
            orders_path = yt.ypath_join(self.workdir, category, 'orders_internal')
            self._delete_rows(order_keys_path, orders_path)

        logging.info('Rows deleting finished')

    def _delete_rows(self, keys_table_path: str, dst_table_path: str) -> None:
        if not self.yt_client.exists(keys_table_path):
            return
        logging.info(f'Deleting rows from {dst_table_path}')
        batches = batch_iterator(
            items=self.yt_client.read_table(keys_table_path),
            batch_size=BATCH_SIZE,
            func=None,
        )
        rows_total = 0
        for batch in batches:
            self.yt_client.delete_rows(dst_table_path, batch)
            rows_total += len(batch)
            logging.info(f'{rows_total} rows deleted')

        logging.info(f'Removing table {keys_table_path}')
        self.yt_client.remove(keys_table_path)
