# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import time
import sys

from yt.wrapper.errors import YtCypressTransactionLockConflict, YtNoSuchTransaction, YtProxyUnavailable

from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_logging import get_logger
from travel.library.python.solomon_push_client import SolomonPushClient
from travel.library.python.yandex_vault import resolve_secrets_ns


LOG = get_logger(__name__)

PYTHON_VERSION = sys.version_info[0]


class MeteredTaskExecutor(object):

    def __init__(self, task_name, options, common_labels):
        self.solomon_client = MeteredTaskExecutor.get_solomon_client(options, common_labels)
        self.task_name = task_name
        self.options = options

    @staticmethod
    def configure(parser):
        parser.add_argument('--vault-token', default=None)

        parser.add_argument("--send-metrics-to-solomon", action="store_true")
        parser.add_argument("--solomon-project", default="travel")
        parser.add_argument("--solomon-service", default="cpa")
        parser.add_argument("--solomon-cluster", default="push_testing")
        parser.add_argument("--solomon-token", default=None)

        parser.add_argument("--fail-fast", action="store_true")

    @staticmethod
    def get_solomon_client(options, common_labels=None):
        if common_labels is None:
            common_labels = dict()
        solomon_client = SolomonPushClient(
            project=options.solomon_project,
            cluster=options.solomon_cluster,
            service=options.solomon_service,
            token=options.solomon_token,
            labels=common_labels,
        )
        return solomon_client

    @staticmethod
    def resolve_secrets(options, common_labels=None):
        if options.vault_token is None:
            LOG.info('No vault token set. Skipping secrets resolving')
            return
        try:
            resolve_secrets_ns(options)
        except Exception:
            LOG.exception('Failed to resolve secrets')
            MeteredTaskExecutor.send_errors(options, common_labels, ErrorType.ET_VAULT_CONNECTION)
            sys.exit(1)

    @staticmethod
    def send_errors(options, common_labels, error_type):
        solomon_client = MeteredTaskExecutor.get_solomon_client(options, common_labels)
        error_metrics = MeteredTaskExecutor.get_error_metrics(error_type)
        MeteredTaskExecutor.send_dict_metrics(solomon_client, error_metrics)

    @staticmethod
    def send_dict_metrics(solomon_client, metrics):
        for key, value in metrics.items():
            solomon_client.send(key, value)

    def try_execute(self):
        pass

    @staticmethod
    def get_error_metrics(error_type):
        error_metrics = dict()
        for sensor in ErrorType:
            value = 1 if error_type == sensor else 0
            error_metrics[sensor.value] = value
        return error_metrics

    def execute(self):
        started = time.time()
        is_successful_run = False

        error_type = None
        exception = None

        if self.options.fail_fast:
            self.try_execute()
        else:
            try:
                self.try_execute()
                is_successful_run = True
            except YtCypressTransactionLockConflict:
                error_type = ErrorType.ET_YT_LOCK
                LOG.exception('Yt lock error')
            except YtNoSuchTransaction:
                error_type = ErrorType.ET_YT_PING
                LOG.exception('Yt ping error')
            except YtProxyUnavailable:
                error_type = ErrorType.ET_YT_PROXY_UNAVAILABLE
                LOG.exception('Yt proxy unavailable error')
            except Exception as exc:
                error_type = ErrorType.ET_UNKNOWN
                if isinstance(exc, ProcessError):
                    error_type = exc.error_type
                else:
                    exception = exc
                    LOG.exception('Unknown exception of type: %s', type(exc))

        delta = time.time() - started
        delta_min = int(delta / 60)
        delta_sec = delta - delta_min * 60
        LOG.info('Task work time {}m {:.2f}s'.format(delta_min, delta_sec))

        exec_metrics = self.get_error_metrics(error_type)
        exec_metrics[self.task_name + '.successful_run'] = 1 if is_successful_run else 0
        exec_metrics[self.task_name + '.failed_run'] = 0 if is_successful_run else 1
        exec_metrics[self.task_name + '.processing_time'] = delta
        LOG.info('Exec metrics: %r', exec_metrics)
        self.send_dict_metrics(self.solomon_client, exec_metrics)

        if self.options.send_metrics_to_solomon:
            self.solomon_client.upload()
            LOG.info('Metrics sent to solomon')

        if error_type == ErrorType.ET_UNKNOWN:
            if PYTHON_VERSION == 2:
                raise
            else:
                raise exception

        LOG.info('All done')
